#!/usr/bin/env python
# -*- coding: utf-8 -*-

# build helpers

import argparse

from utils_layout import RACK_TABLE_BASE, build_racktable_request, check_file, download_and_save, find_and_remove_useless, load_data_from_sources


DEFAULT_OUTPUT_JSON_FNAME = "ipreg-layout.json"

NETS_FILE = "all.networks.txt"
TURBONETS_FILE = "turbonets.txt"
USERSNETS_FILE = "usersnets.txt"


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--layout-json',
                        default=DEFAULT_OUTPUT_JSON_FNAME,
                        help="path to file with result")
    args = parser.parse_args()
    return args


def get_turbo_nets():
    TURBO_NETS_URL = "http://hbf.yandex.net/macros/_TRBOSRVNETS_?format=text&trypo_format=trypo"
    download_and_save(TURBO_NETS_URL, TURBONETS_FILE + ".orig")


def get_users_nets():
    sources_urls = [
        build_racktable_request("ipv4", "usernets"),
        build_racktable_request("ipv6", "usernets")
    ]

    load_data_from_sources(sources_urls, USERSNETS_FILE + ".orig")


def get_networks_from_noc_data():
    sources_urls = [
        build_racktable_request("ipv4", "net_places"),
        build_racktable_request("ipv4", "allocs"),
        build_racktable_request("ipv6", "net_places"),
        build_racktable_request("ipv6", "allocs")
    ]

    load_data_from_sources(sources_urls, NETS_FILE)


def check_resources():
    min_fsize_in_bytes = 10240
    check_file(NETS_FILE, min_fsize_in_bytes)
    check_file(USERSNETS_FILE, min_fsize_in_bytes)

    min_turbo_fsize_in_bytes = 11  # as minimal length of short net: 1.1.1.1/32 (but not ::1/128)
    check_file(TURBONETS_FILE, min_turbo_fsize_in_bytes)


def get_input_data():
    get_networks_from_noc_data()
    get_turbo_nets()
    get_users_nets()

    find_and_remove_useless(TURBONETS_FILE + ".orig", NETS_FILE, TURBONETS_FILE)
    find_and_remove_useless(USERSNETS_FILE + ".orig", NETS_FILE, USERSNETS_FILE)

    check_resources()


def generate_layout(out_json):
    import gen_layout
    gen_layout.generate_layout(
        NETS_FILE,
        TURBONETS_FILE,
        USERSNETS_FILE,
        out_json
    )


def check_layout(out_json):
    min_fsize_in_bytes = 1024 ** 2
    check_file(out_json, min_fsize_in_bytes)

    import test_layout
    result = test_layout.run_full_test(
        out_json,
        NETS_FILE,
        TURBONETS_FILE,
        USERSNETS_FILE,
        ""
    )

    if result != test_layout.RESULT_OK:
        raise Exception("problems were detected during testing")


def make_build(out_json=DEFAULT_OUTPUT_JSON_FNAME):
    get_input_data()
    generate_layout(out_json)
    check_layout(out_json)

    return out_json


if __name__ == "__main__":
    args = parse_args()
    make_build(args.layout_json)
