#!/usr/bin/env python
# -*- coding: utf-8 -*-

from sandbox import sdk2

import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
from sandbox.sdk2.helpers import ProcessLog
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types import client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.errors as ce

from sandbox.projects.common.ya_deploy import release_integration

import datetime
import logging
import os


class CommonBuilderTask(sdk2.Task, release_integration.ReleaseToNannyAndYaDeployTask2):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_XENIAL
        dns = ctm.DnsType.DNS64
        ramdrive = None
        privileged = True
        environments = (environments.PipEnvironment("yandex-yt"),)

    class Parameters(sdk2.Task.Parameters):
        max_restarts = 5

        default_desc = "$(geobase-data builder)"
        description = default_desc

        ramdrive_size = 0

        # with sdk2.parameters.Group("RELEASE-params") as data_release_params:
        datafile_type = "(datafile-type)"
        release_type = "(testing|stable)"
        resource_ttl = "inf"
        result_datafile = "(filename)"

        # with sdk2.parameters.Group("YT-params") as yt_params:
        yt_path_prefix = "//home/geobase/"
        yt_locke_cluster = sdk2.parameters.String('YT-cluster for datafile-announce', required=False, default="locke.yt.yandex.net")
        yt_locke_token_vault_name = sdk2.parameters.String('YT-Loke token vault-name', required=False, default="GEOBASE_YT_LOCKE_ANNOUNCE_TOKEN")
        yt_cluster = sdk2.parameters.String('YT-cluster with some external data', required=True, default="hahn")

        # with sdk2.parameters.Group("token-params") as yt_params:
        yt_token_vault_name = sdk2.parameters.String('YT-token vault-name', required=True, default="GEOBASE_ROBOT_YT_TOKEN")
        yql_token_vault_name = sdk2.parameters.String('YQL-token vault-name', required=True, default="GEOBASE_ROBOT_YQL_TOKEN")
        sandbox_token_vault_name = sdk2.parameters.String('Sandbox-token vault-name', required=True, default="GEOBASE_ROBOT_SANDBOX_TOKEN")
        solomon_token_vault_name = sdk2.parameters.String('Solomon-token vault-name', required=True, default="GEOBASE_ROBOT_SOLOMON_TOKEN")

        # with sdk2.parameters.Group("S3-MDS") as yt_params:
        s3mds_end_point = sdk2.parameters.String('S3 end point', required=True, default="http://s3.mdst.yandex.net") # test; for prod - http://s3.mds.yandex.net
        s3mds_acc_key = sdk2.parameters.String('S3 access key', required=True, default="GEOBASE_S3MDS_DIEASH_ACC_KEY_TESTING")
        s3mds_sec_key = sdk2.parameters.String('S3 secret access key', required=True, default="GEOBASE_S3MDS_DIEASH_SEC_KEY_TESTING")
        s3mds_upload = sdk2.parameters.Bool('Upload to S3-MDS?', required=True, default=False)

        # with sdk2.parameters.Group("RDBMS-params") as db_params:
        db_type = sdk2.parameters.String('db type', required=True, default="mysql")
        db_host = sdk2.parameters.String('db host', required=True, default="geodb-test-01.haze.yandex.net")
        db_port = sdk2.parameters.Integer('db port', required=True, default=3306)
        db_name = sdk2.parameters.String('db name', required=True, default="geodb")
        db_user = sdk2.parameters.String('db user', required=True, default="geo_ro")
        db_pswd = sdk2.parameters.String('db password vault-name', required=True, default="GEOBASE_MYSQL__GEO_RO_PASSWORD")

        add_flags = sdk2.parameters.String('add_flags', required=False, default="")

    class Context(sdk2.Task.Context):
        do_release = False
        rbtorrent = None
        releaser_task_id = None
        has_new_resource = False
        resource_file = None
        new_resource_id = None

    @staticmethod
    def check_file(fname):
        if not os.path.exists(fname):
            raise Exception("no file [%s]" % fname)

        MIN_FILE_SIZE = 100
        fsize = os.path.getsize(fname)
        if fsize < MIN_FILE_SIZE:
            raise Exception("file too small [%s / %s]" % (fname, fsize))

    def run_cmd(self, cmd, desc):
        with ProcessLog(self, logger=logging.getLogger(desc)) as process_log:
            status = sp.Popen(
                cmd,
                shell=True,
                stdout=process_log.stdout,
                stderr=process_log.stdout,
                cwd=os.getcwd()
            ).wait()

        if status != 0:
            err_msg = "cmd[%s] => failed with %s" % (cmd, status)
            raise ce.TaskError(err_msg)

    def check_rbtorrent(self, torrent):
        torrent_len = 50
        torrent_prefix = "rbtorrent:"

        if torrent_len != len(torrent):
            raise Exception("%d != %d in len('%s')" % (torrent_len, len(torrent), torrent))

        if not torrent.startswith(torrent_prefix):
            raise Exception("no required prefix '%s' in [%s]" % (torrent_prefix, torrent))

    def update_yt_node(self, value):
        from yt.wrapper import YtClient

        if not self.Parameters.yt_locke_cluster:
            print ">>> NB: no YT-cluster for announcing..."
            return

        yt_token = sdk2.Vault.data(self.Parameters.yt_locke_token_vault_name)
        yt_client = YtClient(self.Parameters.yt_locke_cluster, token=yt_token)
        node_path = "%s%s-%s" % (self.Parameters.yt_path_prefix, self.Parameters.datafile_type, self.Parameters.release_type)

        self.set_info('announce resource via //%s[%s]' % (self.Parameters.yt_locke_cluster, node_path))

        import time
        if not yt_client.exists(node_path):
            print ">>> NB: YT-node [%s] does not exists! try to create..." % node_path
            yt_client.create(
                type='string_node',
                path=node_path,
                attributes={'ts_created': int(time.time())}
            )

        yt_client.set(node_path, value)
        yt_client.set_attribute(node_path, 'ts_update', int(time.time()))

    def announce_resource(self, rbtorrent):
        if not rbtorrent:
            raise Exception("no rbtorrent")

        if not self.Parameters.yt_locke_token_vault_name:
            print ">>> NB: no YT-node for announcing..."
            return

        self.set_info('trying to announce the resource [%s]' % rbtorrent)
        self.check_rbtorrent(rbtorrent)
        self.update_yt_node(rbtorrent)

    def get_own_resource_type(self):
        return None

    def get_own_type(self):
        return None

    def prepare_resource(self, fname):
        self.set_info('creating resource for [%s]' % fname)
        self.Parameters.description = "%s; %s" % (self.Parameters.description, datetime.datetime.now())

        own_res_type = self.get_own_resource_type()
        resource = own_res_type(
            self,
            description=self.Parameters.description,
            path=fname,
            ttl=self.Parameters.resource_ttl,
            released=self.Parameters.release_type
        )
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()

        self.Context.has_new_resource = True
        self.Context.new_resource_id = resource.id
        logging.info("new_res_id  = %s", self.Context.new_resource_id)

    def prepare_db_credentials(self):
        db_traits_fname = "db.traits"
        db_traits_path = str(self.ramdrive.path / db_traits_fname)

        self.set_info('prepare_db_credentials => [%s]' % db_traits_path)

        with open(db_traits_path, "w") as traits:
            print >>traits, 'db_type="%s"' % self.Parameters.db_type
            print >>traits, 'db_host="%s"' % self.Parameters.db_host
            print >>traits, 'db_name="%s"' % self.Parameters.db_name
            print >>traits, 'db_user="%s"' % self.Parameters.db_user
            print >>traits, 'db_pswd="%s"' % sdk2.Vault.data(self.Parameters.db_pswd)
            print >>traits, 'db_port=%d' % self.Parameters.db_port

    def prepare_yt_credentials(self):
        yt_traits_fname = "yt.traits"
        yt_traits_path = str(self.ramdrive.path / yt_traits_fname)

        self.set_info('prepare_yt_credentials => [%s]' % yt_traits_path)

        with open(yt_traits_path, "w") as traits:
            print >>traits, 'yt_proxy="%s"' % self.Parameters.yt_cluster
            print >>traits, 'yt_token="%s"' % sdk2.Vault.data(self.Parameters.yt_token_vault_name)

    def prepare_tokens_data(self):
        traits_fname = "tokens.list"
        traits_path = str(self.ramdrive.path / traits_fname)

        self.set_info('prepare_token_list => [%s]' % traits_path)

        with open(traits_path, "w") as traits:
            print >>traits, 'export YT_TOKEN="%s"' % sdk2.Vault.data(self.Parameters.yt_token_vault_name)
            print >>traits, 'export YQL_TOKEN="%s"' % sdk2.Vault.data(self.Parameters.yql_token_vault_name)
            print >>traits, 'export SANDBOX_TOKEN="%s"' % sdk2.Vault.data(self.Parameters.sandbox_token_vault_name)
            print >>traits, 'export SOLOMON_TOKEN="%s"' % sdk2.Vault.data(self.Parameters.solomon_token_vault_name)

    def prepare_s3mds_credentials(self):
        traits_fname = "s3mds.credentials"
        traits_path = str(self.ramdrive.path / traits_fname)

        self.set_info('prepare_s3mds_credentials => [%s]' % traits_path)

        with open(traits_path, "w") as traits:
            print >>traits, '[default]'
            print >>traits, 'aws_access_key_id = %s' % sdk2.Vault.data(self.Parameters.s3mds_acc_key)
            print >>traits, 'aws_secret_access_key = %s' % sdk2.Vault.data(self.Parameters.s3mds_sec_key)

    def prepare_add_flags_config(self):
        flags_fname = "add_flags.config"
        flags_path = str(self.ramdrive.path / flags_fname)

        self.set_info('prepare_add_flags_config => [%s]' % flags_path)

        with open(flags_path, "w") as flags_cfg:
            parts = self.Parameters.add_flags.split(' ')
            for param in parts:
                print >>flags_cfg, '%s' % param

    def build(self):
        self.set_info('build datafile => [%s]...' % self.Parameters.result_datafile)

        builder_script = "%s-%s.sh" % (self.Parameters.datafile_type, self.Parameters.release_type)
        bin_path = os.path.dirname(__file__)
        build_util_path = os.path.join(bin_path, "..", "common", builder_script)

        ramdrive_dir = str(self.ramdrive.path) if self.Parameters.ramdrive_size else ""
        curr_dir = "$(pwd)"

        s3mds_env_var = "S3MDS_ENDPOINT=%s" % self.Parameters.s3mds_end_point if self.Parameters.s3mds_upload and self.Parameters.s3mds_end_point else ""

        build_cmd = 'RDBMS_KIND="%s" %s RESULT_FILE="%s" CURR_DIR_PATH="%s" RAM_DRIVE_PATH="%s" /bin/bash %s' % \
                    (self.Parameters.db_type, s3mds_env_var, self.Parameters.result_datafile, curr_dir, ramdrive_dir, build_util_path)
        self.run_cmd(build_cmd, 'build-datafile [%s]' % self.Parameters.result_datafile)

        if self.Parameters.result_datafile:
            CommonBuilderTask.check_file(self.Parameters.result_datafile)
        return self.Parameters.result_datafile

    def get_rbtorrent(self):
        self.set_info('get_rbtorrent...')

        if not self.Context.new_resource_id:
            raise Exception("no resource-id")

        resource = sdk2.Resource.find(task=self, id=self.Context.new_resource_id).first()
        self.Context.rbtorrent = resource.skynet_id
        logging.info("rbtorrent(skynet_id) := %s" % self.Context.rbtorrent)

    def run_builder_and_releaser(self):
        self.set_info('run_builder()... task#%s' % self.id)

        self.prepare_db_credentials()
        self.prepare_yt_credentials()
        self.prepare_s3mds_credentials()
        self.prepare_add_flags_config()
        self.prepare_tokens_data()
        self.Context.resource_file = self.build()

        if not self.get_own_resource_type():
            self.set_info('NB: without resource. task#%s' % self.id)
            return

        if not self.Context.resource_file:
            raise Exception("no resource file")

        self.prepare_resource(self.Context.resource_file)
        self.mark_released_resources(self.Parameters.release_type)

        if self.Context.releaser_task_id:
            return

        self.get_rbtorrent()

        own_type = self.get_own_type()
        releaser_task = own_type(
            parent=self,
            do_release=True,
            description="auto-release of %s" % self.Parameters.description
        )
        logging.info("starting releaser-task#%s" % releaser_task.id)
        self.Context.releaser_task_id = releaser_task.id
        releaser_task.enqueue()

    def wait_builder_and_make_release(self):
        self.set_info('wait_builder_and_make_release()...  task#%s' % self.id)

        builder_task = self.parent
        with self.memoize_stage.wait_for_builder:
            logging.info("waiting data from main builder - task#%s" % builder_task.id)
            raise sdk2.WaitTask(
                tasks=[builder_task],
                statuses=(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True)

        logging.info("builder process finished - task#%s / status %s (wanted SUCCESS %s)"
                     % (builder_task.id, builder_task.status, ctt.Status.SUCCESS))
        logging.info("res_id [%s] / sky_id [%s] / desc [%s]"
                     % (builder_task.Context.new_resource_id, builder_task.Context.rbtorrent, builder_task.Parameters.description))

        if builder_task.status != ctt.Status.SUCCESS:
            return

        if not builder_task.Context.rbtorrent:
            raise Exception("no skynet_id")

        self.server.release(
            task_id=builder_task.id,
            type=builder_task.Parameters.release_type,
            subject=builder_task.Parameters.description
        )
        self.announce_resource(builder_task.Context.rbtorrent)

    def on_enqueue(self):
        if self.Parameters.ramdrive_size:
            self.Requirements.ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, self.Parameters.ramdrive_size * 1024, None)

    def on_execute(self):
        if self.Context.do_release:
            self.wait_builder_and_make_release()
        else:
            self.run_builder_and_releaser()

    def on_release(self, additional_parameters):
        release_integration.ReleaseToNannyAndYaDeployTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
