#!/usr/bin/env python
# coding=utf-8

from __future__ import print_function

import sys
import json


BAD_ID = -5
ROOT_ID = 0
EARTH_ID = 10000
RUSSIA_ID = 225
DEFAULT_ROOT_ID = EARTH_ID


def parse_args():
    import argparse
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--geodata',
                        default="geodata.bin",
                        help="path to geodata for checking")
    parser.add_argument('--root-id',
                        type=int,
                        default=DEFAULT_ROOT_ID,
                        help="id of tree root")
    parser.add_argument('--unicode-escape',
                        action='store_true',
                        help="unicode-escape in string values")
    parser.add_argument('--fields-list',
                        default="",
                        help="comma-separated fields list for export; default - empty for all")
    parser.add_argument('--table-data',
                        required=True,
                        help="path to file for data")
    parser.add_argument('--table-scheme',
                        required=True,
                        help="path to file for scheme")
    args = parser.parse_args()
    return args


stats = {}


def inc_stat_counter(counter_name, value=1):
    stats[counter_name] = stats.get(counter_name, 0) + value


useless_fields_list = [
    'position',
    'bgp_name',
    'phone_code_old',
    'zip_code_old',
    'services',
    'suggest_list',
]


def make_services_as_attrs(full_srv_list, reg_srv_list, wanted_fields, result):
    selected_services = full_srv_list if not wanted_fields else [x for x in full_srv_list if x in wanted_fields]

    for srv in selected_services:
        result[srv] = srv in reg_srv_list


def make_linguistics_as_attrs(lookup, reg_id, wanted_fields, result):
    for lang in lookup.get_known_linguistics():
        ling_cases = lookup.get_linguistics(reg_id, lang)
        for case in ling_cases.GetAllNames():
            full_case = "%s_%s" % (lang, case)
            if not wanted_fields or full_case in wanted_fields:
                result[full_case] = getattr(ling_cases, case)


def get_type(value):
    if isinstance(value, bool):
        return 'boolean'
    elif isinstance(value, int):
        return 'int32'
    elif isinstance(value, str):
        return 'string'
    elif isinstance(value, float):
        return 'double'
    else:
        return 'any'


def get_scheme_from_attrs(json_attrs):
    scheme_columns = []
    for f_name, value in sorted(json_attrs.iteritems()):
        f_type = get_type(value)
        f_opts = "required=" + ("false" if 'any' == f_type else "true")

        if 'reg_id' == f_name:
            f_opts += ";sort_order=ascending"

        col = "{name=%s;type=%s;%s}" % (f_name, f_type, f_opts)
        if 'reg_id' == f_name:
            scheme_columns = [col] + scheme_columns
        else:
            scheme_columns.append(col)

    return scheme_columns


def get_regions(geodata, root_id=DEFAULT_ROOT_ID, fields_list="", unicode_escape=False):
    import geobase6

    wanted_fields = [] if not fields_list else fields_list.split(',')
    lookup = geobase6.Lookup(geodata)

    regs_list = lookup.get_tree(root_id)
    regs_sorted_list = sorted(regs_list, key=lambda r: r['id'])

    parents_list_add = True if not fields_list else "parents_list" in wanted_fields
    children_list_add = True if not fields_list else "children_list" in wanted_fields

    country_id_add = True if not fields_list else "country_id" in wanted_fields
    country_iso_add = True if not fields_list else "country_iso" in wanted_fields

    enabled_fields = wanted_fields if wanted_fields else [f for f in regs_sorted_list[0].keys() if f not in useless_fields_list]
    resulted_regs = []
    for reg in regs_sorted_list:
        inc_stat_counter('regs')
        reg_id = reg['id']
        final_reg = {k: reg[k] for k in enabled_fields}
        final_reg['reg_id'] = reg_id
        del final_reg['id']

        make_services_as_attrs(geobase6.get_known_services(), reg['services_names'], wanted_fields, final_reg)
        make_linguistics_as_attrs(lookup, reg_id, wanted_fields, final_reg)

        if parents_list_add:
            final_reg['parents_ids'] = lookup.get_parents_ids(reg_id)

        if children_list_add:
            final_reg['children_ids'] = lookup.get_children_ids(reg_id)

        if country_id_add and country_iso_add:
            country_id = 0
            country_iso = ""

            if reg_id > 0 and -1 != reg['type'] and -1 != reg['parent_id']:
                inc_stat_counter('regs_with_country')

                country_id = lookup.get_country_id(reg_id)
                country_reg = lookup.get_region_by_id(country_id)
                country_iso = country_reg['iso_name']

            if not country_id:
                inc_stat_counter('regs_country_unknown')
            if not country_iso:
                inc_stat_counter('country_empty_iso')

            final_reg['country_id'] = country_id
            final_reg['country_iso'] = country_iso

        resulted_regs.append(final_reg)

    return resulted_regs


if __name__ == "__main__":
    retval = 0
    try:
        args = parse_args()
        regions = get_regions(
            geodata=args.geodata,
            root_id=args.root_id,
            fields_list=args.fields_list,
            unicode_escape=args.unicode_escape,
        )

        tbl_data_stream = open(args.table_data, "w")
        tbl_scheme_stream = open(args.table_scheme, "w")

        for json_attrs in regions:
            print(json.dumps(json_attrs, separators=(',', ':')), file=tbl_data_stream)

        scheme_columns = get_scheme_from_attrs(regions[-1])
        print("<schema=[%s]>" % ';'.join(scheme_columns), file=tbl_scheme_stream)
    except Exception as ex:
        print("// ex: %s" % ex, file=sys.stderr)
        retval = 1

    print(json.dumps(stats, sort_keys=True, indent=4, separators=(',', ':')), file=sys.stderr)
    sys.exit(retval)
