#!/usr/bin/env python
# -*- coding: utf-8 -*-

# simple pipe-convector from 'IP*Layout.txt' to 'layout.json' for 'libipreg1'

import argparse
import datetime
import ipaddr
import sys


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--nets-list',
                        default="all.networks.txt",
                        help="path to all YA networks")
    parser.add_argument('--turbonets-list',
                        default="turbonets.txt",
                        help="path to torbo networks")
    parser.add_argument('--usersnets-list',
                        default="usersnets.txt",
                        help="path to user networks")
    parser.add_argument('--layout-json',
                        default="",
                        help="path to output file; 'stdout' will be used if empty")
    args = parser.parse_args()
    return args


def is_ipv6(addr):
    return -1 != addr.find(':')


def to_ipv6(addr):
    return addr if is_ipv6(addr) else '::ffff:' + addr


def parent_net(net, nets):
    return next((x for x in nets if net in x), None)


def aggregate_exists(net, nets):
    return None != parent_net(net, nets)


def normalize_networks(filename):
    nets = []

    for line in open(filename, 'r'):
        line = line.strip()

        net = ipaddr.IPv6Network(line) if is_ipv6(line) else ipaddr.IPv4Network(line)
        if 1 == net.numhosts:
            print >>sys.stderr, "%s: %s was skipped" % (filename, line)
            continue
        if aggregate_exists(net, nets):
            continue

        nets.append(net)
        nets = [x for x in nets if not (x in net and net != x)]

    return nets


class Builder(object):
    def __init__(self, networks, turbo_nets, users_nets, layout_json):
        self.networks_fname = networks

        self.networks = normalize_networks(networks)
        self.turbo_nets = normalize_networks(turbo_nets)
        self.users_nets = normalize_networks(users_nets)

        self.layout_json = open(layout_json, "w") if layout_json else sys.stdout

    def calc_flags(self, net):
        flags = []

        parent = parent_net(net, self.turbo_nets)
        if parent:
            flags.append('\"turbo\": true')
            flags.append('\"trusted-proxy\": true')  # TODO(dieash@) obsolete; must be removed
            if net != parent:
                flags.append('\"t-sub\": true')

        parent = parent_net(net, self.users_nets)
        if parent:
            flags.append('\"user\": true')
            if net != parent:
                flags.append('\"u-sub\": true')

        parent = parent_net(net, self.networks)
        if parent and net != parent:
            flags.append('\"sub\": true')

        if not flags:
            return ''

        return ',"flags": {%s}' % ','.join([str(x) for x in flags])

    def process_net(self, net_str):
        net_str = net_str.strip()
        net = ipaddr.IPv6Network(net_str) if is_ipv6(net_str) else ipaddr.IPv4Network(net_str)

                       # TODO(dieash@) uncomment if you want to see IPv6 only
        low = net[0]  # if is_ipv6(net_str) else str(ipaddr.IPv6Address(to_ipv6(str(net[0]))))
        high = net[-1]  # if is_ipv6(net_str) else str(ipaddr.IPv6Address(to_ipv6(str(net[-1]))))
        flags = self.calc_flags(net)

        return '{"low":"%s","high":"%s"%s}' % (low, high, flags)

    def generate(self):
        nets_file = open(self.networks_fname, 'r')

        start_time = ',"build_time_tag":"' + str(datetime.datetime.now()) + '"}'
        first_line = self.process_net(nets_file.readline())
        first_line = first_line.replace('}', start_time)

        self.layout_json.write('[' + first_line)
        for line in nets_file:
            self.layout_json.write('\n,' + self.process_net(line))
        self.layout_json.write(']\n')


def generate_layout(networks, turbo_nets, users_nets, layout_json=""):
    builder = Builder(networks, turbo_nets, users_nets, layout_json)
    builder.generate()


def main(args):
    generate_layout(args.nets_list, args.turbonets_list, args.usersnets_list, args.layout_json)


if __name__ == "__main__":
    main(parse_args())
