#!/bin/bash -E
set -aeo pipefail

script_dir="$(dirname $(readlink -f $0))"
bin_dir=${script_dir}

data_dir=${1:-"."}
if [ -n "${data_dir}" ]; then
    mkdir --parents ${data_dir}
fi

racktable_base_url="https://racktables.yandex.net/export"
sandbox_base="https://proxy.sandbox.yandex-team.ru"

simple_log() {
    echo "$(date +%Y%m%d.%H%M%S) $*" >&2
}

download_file() {
    simple_log ${FUNCNAME} $*

    local file_url=$1
    curl --insecure --silent --fail --show-error --retry 5 --location "${file_url}"
}

download_yt_ipreg_patch() {
    simple_log ${FUNCNAME} $*

    check_file ${data_dir}/yt.traits
    . ${data_dir}/yt.traits

    local dsv_format="<columns=[key;value];missing_value_mode=print_sentinel;enable_escaping=false>schemaful_dsv"
    local patch_yt_tbl=$1
    local out_fname=$2
    YT_PROXY=${yt_proxy} YT_TOKEN=${yt_token} yt read "${patch_yt_tbl}" --format "${dsv_format}" | tee ${out_fname} | wc -l
}

build_racktable_request() {
    local report_name=$1
    local ip_kind=$2

    local request_url="${racktable_base_url}/networklist.php?only=${ip_kind}&report=${report_name}"
    echo ${request_url}
}

get_turbo_nets() {
    simple_log ${FUNCNAME} $*

    local turbonets_url="http://hbf.yandex.net/macros/_TRBOSRVNETS_?format=text&trypo_format=trypo"
    download_file ${turbonets_url}
}

get_users_nets() {
    simple_log ${FUNCNAME} $*

    download_file $(build_racktable_request "usernets" "ipv4")
    download_file $(build_racktable_request "usernets" "ipv6")
}

get_yandex_nets() {
    simple_log ${FUNCNAME} $*

    download_file $(build_racktable_request "net_places" "ipv4")
    download_file $(build_racktable_request "allocs" "ipv4")

    download_file $(build_racktable_request "net_places" "ipv6")
    download_file $(build_racktable_request "allocs" "ipv6")
}

networks_fname="${data_dir}/yandex.list"
users_nets_fname="${data_dir}/users.list"
turbo_nets_fname="${data_dir}/turbo.list"

check_noc_data() {
    test -s ${networks_fname}
    test -s ${users_nets_fname}
    test -s ${turbo_nets_fname}
}

download_noc_data() {
    simple_log ${FUNCNAME} $*

    get_yandex_nets > ${networks_fname}
    get_users_nets  > ${users_nets_fname}

    get_turbo_nets \
    | tee {turbo_nets_fname}.orig \
    | ${script_dir}/utils_proj_nets.py \
    > ${turbo_nets_fname}
}

make_whole_addr_space() {
    simple_log ${FUNCNAME} $*

    local mark=$1
    cat <<- EOF_IP
	::-ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff	{"${mark}":1}
	EOF_IP
}

prepare_yandex_patch() {
    simple_log ${FUNCNAME} $*

    local inet_mark="__inet__"
    make_whole_addr_space ${inet_mark} > ${data_dir}/full_addr_space.list

    for fname in ${networks_fname} ${users_nets_fname} ${turbo_nets_fname}; do
        local base_fname=$(basename ${fname})
        local datakind_mark=${base_fname%%.list}

        cat $fname \
        | awk '{ print $1 }' \
        | ${bin_dir}/ipreg-fmt --empty-data --add-json-attr ${datakind_mark},reliable \
        | sort \
        | ${bin_dir}/ipreg-split -u \
        > ${fname}.patch
    done

    cat ${data_dir}/full_addr_space.list \
    | ${bin_dir}/ipreg-patch --patch-data ${networks_fname}.patch \
    | ${bin_dir}/ipreg-patch --patch-data ${users_nets_fname}.patch \
    | ${bin_dir}/ipreg-patch --patch-data ${turbo_nets_fname}.patch \
    | ${bin_dir}/ipreg-merge \
    | ${bin_dir}/ipreg-check-addr-space --strict-monotonic \
    | ${bin_dir}/ipreg-fmt --remove-json-attr ${inet_mark} \
    | grep -v "{}$"
}

get_ipreg_data_url() {
    local wanted_path="$1"
    local sbr_traits="?owner=IPREG&attrs=%7B%22released%22:%22stable%22%7D"
    echo "${sandbox_base}/${wanted_path}${sbr_traits}"
}

get_geodata_url() {
    local sbr_traits="?owner=GEOBASE&attrs=%7B%22released%22:%22stable%22%7D"
    echo "${sandbox_base}/last/GEODATA6BIN_STABLE${sbr_traits}"
}

get_maxmind_resource_url() {
    local required_fname=$1
    local maxmind_traits="?owner=IPREG&attrs=%7B%22released%22:%22testing%22,%22source%22:%22maxmind%22%7D"

    echo "${sandbox_base}/last/IPREG_DATA/${required_fname}${maxmind_traits}"
}

prepare_maxmind_patch_common() {
    simple_log ${FUNCNAME} $*

    local mm_mark="$1"
    local mm_processor="$2"

    local mm_fname="${mm_mark}-CSV.zip"
    if [ ! -e ${mm_fname} ]; then
        download_file $(get_maxmind_resource_url ${mm_fname}) > ${data_dir}/${mm_fname}
    fi

    local v4_data_fname="${data_dir}/${mm_mark}-Blocks-IPv4.csv"
    local v6_data_fname="${data_dir}/${mm_mark}-Blocks-IPv6.csv"

    if [ ! -e ${v4_data_fname} -a ! -e ${v6_data_fname} ]; then
        unzip -oj ${data_dir}/${mm_fname} -d ${data_dir} ${mm_mark}*/*.csv 1>&2
    fi
    test -s ${v4_data_fname}
    test -s ${v6_data_fname}

    cat ${v4_data_fname} ${v6_data_fname} \
    | ${bin_dir}/${mm_processor} \
    | ${bin_dir}/ipreg-merge
}

prepare_maxmind_anon_patch() {
    local mm_mark="GeoIP2-Anonymous-IP"
    local mm_processor="maxmind_anon_data_processor.py --mode=convert"
    local out_fname=$1

    prepare_maxmind_patch_common "${mm_mark}" "${mm_processor}" > ${out_fname}
}

prepare_maxmind_isp_patch() {
    local out_fname=$1
    local names_fname="${out_fname}.names"

    local mm_mark="GeoIP2-ISP"
    local mm_processor="maxmind_isp_data_processor.py"

    prepare_maxmind_patch_common "${mm_mark}" "${mm_processor} --no-subst" > ${out_fname}_w_names
    prepare_maxmind_patch_common "${mm_mark}" "${mm_processor} --orgs-output ${names_fname}.tmp" > ${out_fname}
    cat ${names_fname}.tmp | sort -k1n | tr '[:upper:]' '[:lower:]' > ${names_fname}
    rm ${names_fname}.tmp
}

apply_patch() {
    simple_log ${FUNCNAME} $*

    local patch_fname=$1
    test -s ${patch_fname}

    ${bin_dir}/ipreg-patch --patch-data ${patch_fname} \
    | ${bin_dir}/ipreg-check-addr-space --completeness
}

make_ipreg_compacted() {
    simple_log ${FUNCNAME} $*

    local geodata_path=${1:-"/var/cache/geobase/geodata6.bin"}

    ${bin_dir}/ipreg-fmt --remove-json-attr reliability \
    | ${bin_dir}/ipreg-merge

# NB: disabled because diff by region_id for ranges were detected during tests
#
#    | ${bin_dir}/ipreg-merge --geodata ${geodata_path} --subst-middle-id --by-regions-only --no-clarification \
#    | ${bin_dir}/ipreg-merge
#
}

group_and_sort_desc() {
    sort \
    | uniq -c \
    | sort -k1nr \
    | awk '{ print $2 "\t" $1 }'
}

column1_only() {
    awk '{ print $1 }'
}

join_as_in_list() {
    sed 's/\tAS/,/g' | sed 's/,/\t/'
}

make_json_sorter() {
    cat <<- EOF_SORT
	import sys
	import json

	for line in sys.stdin:
	    ip_range, json_data = line.strip().split('\t')
	    as_sorted_data = sorted([ int(x) for x in json.loads(json_data)['as'].split(',')])
	    print '%s\t{"as":"%s"}' % (ip_range, ",".join([str(x) for x in as_sorted_data]))
	EOF_SORT
}

sort_as_json() {
    make_json_sorter > ${script_dir}/as-json-sorter.py
    python ${script_dir}/as-json-sorter.py
}

add_as_json_attr() {
    sed -e 's:$:"}:' -e 's/\t/\t{"as":"/'
}

del_as_json_attr() {
    sed -e 's:"}$::' -e 's/{"as":"//'
}

prepare_as_section_data() {
    local as_json=$1
    local as_joined=$2

    add_as_json_attr \
    | sort_as_json | tee ${as_json} \
    | del_as_json_attr \
    | tee ${as_joined} \
    | awk '{ $1 = ""; print $0 }' \
    | tr -d ' ' \
    | group_and_sort_desc
}

add_as_attr() {
    sed -e 's:$:}:' -e 's/\t/\t{"as":/'
}

add_dot_reg_id_attr() {
    sed 's:"reg:".reg:'
}

del_dot_reg_id_attr() {
    sed 's:"\.:":'
}

sort_json() {
    ${script_dir}/ipreg-merge --sort-json
}

prepare_traits_section() {
    # NB: we cut prefix for quick comparing in 'replace_traits_by_idx.py' below
    sed 's/^.*_id"://' \
    | tr -d '}' \
    | group_and_sort_desc
}

traits_postproc() {
    sed -e 's:$:}:' -e 's/^/{"region_id":/'
}

prepare_as_yandex_patch() {
    simple_log ${FUNCNAME} $*

    local as_yandex_fname="$1"
    local as_section_fname="$1.list"
    local ipreg_src_url=$2
    download_file $(get_ipreg_data_url ${ipreg_src_url}/IPv6/ip_origin.gz) | zcat > ${as_yandex_fname}.orig
    test -s ${as_yandex_fname}.orig

    cat ${as_yandex_fname}.orig \
    | join_as_in_list \
    | prepare_as_section_data ${as_yandex_fname}.orig.json ${as_yandex_fname}.as_joined \
    | tee ${as_section_fname}.tmp \
    | column1_only \
    > ${as_section_fname}

    cat ${as_yandex_fname}.as_joined \
    | ${script_dir}/replace_traits_by_idx.py --proc asset --traits ${as_section_fname} --debug-counter 10000 \
    | add_as_attr \
    > ${as_yandex_fname}

# check compatibility of MixMind and own data:
# 1. prepare AS-data as json-ipreg
#    $ cat IPREG-as.patch.orig \
#       | sed 's/\tAS/,/g' \
#       | sed 's/,/\t/' \
#       | sed -e 's:$:"}:' -e 's/\t/\t{"as":"/' \
#       > IPREG-as.patch.orig.json
#
# 2. patch the IPREG with Maxmind-data:
#    $ cat IPREG.json_yandex_mmanon_mmisp \
#       | ./ipreg-patch -p IPREG-as.patch.orig.json
#       > IPREG.json_IPREG.json_yandex_mmanon_mmisp__as
#
# 3. extract only required fields and compare them:
#    $ cat IPREG.json_IPREG.json_yandex_mmanon_mmisp__as \
#       | grep -P '\t{"as":' \
#       | ./ipreg-fmt --extract-json-attr as,as_num --no-ip-range \
#       | sed 's:",":\t:' \
#       | tr -d '{}"as_num:' \
#       | awk '$2 != "" && 0 == index($1, $2) { print }' \
#       | tee as-difference_ya-vs-mm.list \
#       | sort | uniq -c  | wc -l # ~2.6K
#                     ... | sort -k1nr | awk '$1 > 100 { print }'
}

get_ipreg_utils_url() {
    local sbr_traits="?owner=GEOBASE"
    echo "${sandbox_base}/2522816173${sbr_traits}"
}

prepare_ipreg_utils() {
    simple_log ${FUNCNAME} $*

    local util_arch="ipreg-utils.tar.gz"
    download_file $(get_ipreg_utils_url) > ${util_arch}

    tar xvf ${util_arch}
    test -s ipreg-fmt
    test -s ipreg-check-addr-space
    test -s ipreg-merge
    test -s ipreg-patch

    cp -t ${bin_dir} ipreg-*
}

check_substitution() {
    local orig_file=$1
    local new_file=$2

    local id_from=$3
    local id_to=$4

    local id_from_orig=$(grep "_id\":${id_from}}" ${orig_file} | wc -l)
    local id_to_orig=$(grep "_id\":${id_to}}" ${orig_file} | wc -l)

    local id_to_new=$(grep "_id\":${id_to}}" ${new_file} | wc -l)

    echo "${orig_file} count(${id_from}/${id_to}) => ${id_from_orig}/${id_to_orig}; ${new_file} count(${id_to}) => ${id_to_new}"
    if [ $((${id_from_orig} + ${id_to_orig})) -ne ${id_to_new} ]; then
        echo "SUBST MISMATCH"
        exit 1
    fi
}

make_reg_ids_substitution() {
    local regs_ids_subst_list=$1 # NB: REGS_IDS_SUBST := ${reg1_org}:${reg1_new}[,]...
    local traits_fname=$2

    cp ${traits_fname} ${traits_fname}.orig
    for pairs in $(echo -n ${regs_ids_subst_list} | tr ',' ' '); do
        local id_from=${pairs%%:*}
        local id_to=${pairs##*:}

        echo "make subst ${id_from} => ${id_to}..."
        cat ${traits_fname} | sed "s/_id\":${id_from}}/_id\":${id_to}}/" > ${traits_fname}.subst

        check_substitution ${traits_fname} ${traits_fname}.subst ${id_from} ${id_to}
        mv ${traits_fname}.subst ${traits_fname}
    done
}

main_yandex() {
    simple_log ${FUNCNAME}
    prepare_ipreg_utils

    local base_ipreg_fname="${data_dir}/IPREG.json"
    local base_ipgsm_fname="${data_dir}/IPGSM2"
    local merged_ipreg_fname="${base_ipreg_fname}_wo_reliabilities"
    local geodata_fname="${data_dir}/geodata6.bin"

    if [ ! -f ${data_dir}/add_flags.config ]; then
        echo 'IPREG_RES_URL="last/IPREG_EXPORT"' > ${data_dir}/add_flags.config
    fi
    . ${data_dir}/add_flags.config # IPREG_RES_URL / ML_PATCH_RES_URL / ML_PATCH_TRAITS / REGS_IDS_SUBST

    if [ ! -s ${geodata_fname} ]; then
        download_file $(get_geodata_url) > ${geodata_fname}
    fi
    test -s ${geodata_fname}

    if [ ! -s ${base_ipreg_fname} ]; then
        local json_path=${IPREG_JSON_PATH:-"IPv6/IPREG.json"}
        download_file $(get_ipreg_data_url ${IPREG_RES_URL}/${json_path}) > ${base_ipreg_fname}
    fi
    test -s ${base_ipreg_fname}

    if [ ! -s ${base_ipgsm_fname} ]; then
        download_file $(get_ipreg_data_url ${IPREG_RES_URL}/IPv6/IPGSM2) > ${base_ipgsm_fname}
    fi
    test -s ${base_ipgsm_fname}

    if [ -n "${IPV6_NET64_ON}" ]; then
        simple_log "IPv6 networks cutting..."
        simple_log "orig-size $(wc -l ${base_ipreg_fname})"

        cat ${base_ipreg_fname} \
           | ./ipreg-fmt --ipv6-nets-size=64 --ip-format=short --add-orig-size --sort-json \
           | ./ipreg-merge --join-equal-ranges \
           | ./ipreg-fmt --remove-json-attr orig_net_size \
           | ./ipreg-merge \
           | ./ipreg-check-addr-space --strict-monotonic \
           | tee ${base_ipreg_fname}.tmp \
           | wc -l

        simple_log "replace IPREG with new datafile..."
        mv ${base_ipreg_fname}.tmp ${base_ipreg_fname}
    fi

    if [ -n "${IPREGGEO_ON}" ]; then
        simple_log "IPREGGEO patch..."

        local ipgeo_patch_fname="${data_dir}/ipgeo-patch.json"
        local patch_yt_tbl="//home/geotargeting/production/sources/ipreggeo_patch"
        local ipreggeo_yt_tbl="${IPREGGEO_YT_PATCH:-${patch_yt_tbl}}"
        download_yt_ipreg_patch ${ipreggeo_yt_tbl} ${ipgeo_patch_fname}.tmp
        cat ${ipgeo_patch_fname}.tmp | ./ipreg-merge | tee ${ipgeo_patch_fname} | wc -l

        local ipreg_ipgeo_fname="${data_dir}/IPREG+ipgeo.json"
        cat ${base_ipreg_fname} | apply_patch ${ipgeo_patch_fname} | tee ${ipreg_ipgeo_fname} | wc -l

        simple_log "IPREGGEO-patched IPREG subst..."
        rm ${ipgeo_patch_fname} ${ipgeo_patch_fname}.tmp ${base_ipreg_fname}
        base_ipreg_fname=${ipreg_ipgeo_fname}
    fi

    cat ${base_ipreg_fname} | make_ipreg_compacted ${geodata_fname} > ${merged_ipreg_fname}

    if [ -n "${REGS_IDS_SUBST}" ]; then
        make_reg_ids_substitution ${REGS_IDS_SUBST} ${merged_ipreg_fname}
    fi

    local gsm_ipreg_fname="${base_ipreg_fname}_mobile"
    local gsm_patch_fname="${data_dir}/IPREG-mobile.patch"

    cat ${base_ipgsm_fname} | grep -v ^2002: | ${bin_dir}/ipreg-fmt --add-json-attr mobile --extract-json-attr mobile > ${gsm_patch_fname}
    test -s ${gsm_patch_fname}

    cat ${merged_ipreg_fname} | apply_patch ${gsm_patch_fname} > ${gsm_ipreg_fname}
    test -s ${gsm_ipreg_fname}
    # we have to save ${merged_ipreg_fname} for further IP checks after generation

    download_noc_data
    check_noc_data

    local ya_patch_ipreg_fname="${gsm_ipreg_fname}_yandex"
    local ya_patch_fname="${data_dir}/IPREG-yandex.patch"

    prepare_yandex_patch > ${ya_patch_fname}
    cat ${gsm_ipreg_fname} | apply_patch ${ya_patch_fname} > ${ya_patch_ipreg_fname}
    rm ${gsm_ipreg_fname}

    local mm_anon_ipreg_fname="${ya_patch_ipreg_fname}_mmanon"
    local mm_anon_patch_fname="${data_dir}/IPREG-mmanon.patch"

    prepare_maxmind_anon_patch ${mm_anon_patch_fname}
    cat ${ya_patch_ipreg_fname} | apply_patch ${mm_anon_patch_fname} > ${mm_anon_ipreg_fname}
    rm ${ya_patch_ipreg_fname}

    local mm_isp_ipreg_fname="${mm_anon_ipreg_fname}_mmisp"
    local mm_isp_patch_fname="${data_dir}/IPREG-mmisp.patch"

    prepare_maxmind_isp_patch ${mm_isp_patch_fname}
    cat ${mm_anon_ipreg_fname} | apply_patch ${mm_isp_patch_fname} > ${mm_isp_ipreg_fname}
    rm ${mm_anon_ipreg_fname}

    local as_ipreg_fname="${mm_isp_ipreg_fname}_as"
    local as_yandex_patch_fname="${data_dir}/IPREG-as.patch"

    prepare_as_yandex_patch ${as_yandex_patch_fname} ${IPREG_RES_URL} # TODO(dieash@) join MM-as with YA-as list?
    cat ${mm_isp_ipreg_fname} | apply_patch ${as_yandex_patch_fname} > ${as_ipreg_fname}
    rm ${mm_isp_ipreg_fname}

    local hpv_ipreg_fname="${as_ipreg_fname}_hpv"
    local hpv_patch="${data_dir}/hosting+proxy+vpn.patch"
    if [ -n "${HPV_PATCH_ON}" ]; then
        ${script_dir}/LAAS-964.hosting+proxy+vpn-fix/build-patch.sh ${bin_dir} ${mm_anon_patch_fname} ${hpv_patch}
    fi

    if [ -f ${hpv_patch} ]; then
        simple_log "HOSTING/VPN/PROXY patch..."
        cat ${as_ipreg_fname} \
        | ${bin_dir}/ipreg-patch -p ${hpv_patch} \
        | ${bin_dir}/ipreg-merge \
        | ${bin_dir}/ipreg-check-addr-space --completeness \
        > ${hpv_ipreg_fname}.tmp

        mv -f ${as_ipreg_fname} ${as_ipreg_fname}.orig
        mv -f ${hpv_ipreg_fname}.tmp ${as_ipreg_fname}
    fi

    if [ -n "${ICLOUD_PATCH_ON}" ]; then
        simple_log "iCloud patch..."

        local icloud_patch_fname="${data_dir}/icloud-patch.json"
        local patch_yt_tbl="//home/geotargeting/production/icloud_private_relay/patch"
        local icloud_yt_tbl="${ICLOUD_YT_PATCH:-${patch_yt_tbl}}"
        download_yt_ipreg_patch ${icloud_yt_tbl} ${icloud_patch_fname}.tmp

        cat ${icloud_patch_fname}.tmp \
        | ${bin_dir}/ipreg-fmt -f ipv6 \
        | sort \
        | ${bin_dir}/ipreg-split -m --only-final \
        | tee ${icloud_patch_fname} \
        | wc -l

        local ipreg_icloud_fname="${data_dir}/IPREG+icloud.json"
        cat ${as_ipreg_fname} | apply_patch ${icloud_patch_fname} | tee ${ipreg_icloud_fname} | wc -l

        simple_log "iCloud-patched IPREG subst..."
        rm ${icloud_patch_fname} ${icloud_patch_fname}.tmp ${as_ipreg_fname}
        as_ipreg_fname=${ipreg_icloud_fname}
    fi

    local ipreg_final_fname="${data_dir}/IPREG.final"
    local traits_section_fname="${data_dir}/IPREG-traits.list"

    cat ${as_ipreg_fname} \
    | add_dot_reg_id_attr \
    | sort_json \
    | del_dot_reg_id_attr \
    | tee ${as_ipreg_fname}.sort \
    | prepare_traits_section \
    | tee ${traits_section_fname}.tmp \
    | column1_only \
    | traits_postproc \
    > ${traits_section_fname}

    cat ${as_ipreg_fname}.sort \
    | ${script_dir}/replace_traits_by_idx.py --traits ${traits_section_fname}.tmp --debug-counter 250000 \
    | ${script_dir}/ipreg-fmt --split-mix-ranges \
    | ${bin_dir}/ipreg-check-addr-space --completeness \
    > ${ipreg_final_fname}
    rm ${as_ipreg_fname}.sort ${traits_section_fname}.tmp
}

main_eu_only() {
    simple_log ${FUNCNAME}
    prepare_ipreg_utils

    local base_ipreg_fname="${data_dir}/IPREG.json"
    local merged_ipreg_fname="${base_ipreg_fname}_wo_reliabilities"
    local geodata_fname="${data_dir}/geodata6.bin"

    if [ ! -f ${data_dir}/add_flags.config ]; then
        echo 'IPREG_RES_URL="last/IPREG_EXPORT"' > ${data_dir}/add_flags.config
    fi
    . ${data_dir}/add_flags.config # IPREG_RES_URL / ML_PATCH_RES_URL / ML_PATCH_TRAITS / REGS_IDS_SUBST

    if [ ! -s ${geodata_fname} ]; then
        download_file $(get_geodata_url) > ${geodata_fname}
    fi
    test -s ${geodata_fname}

    if [ ! -s ${base_ipreg_fname} ]; then
        local json_path=${IPREG_JSON_PATH:-"IPv6/IPREG.json"}
        download_file $(get_ipreg_data_url ${IPREG_RES_URL}/${json_path}) > ${base_ipreg_fname}

        ls -al ${base_ipreg_fname}
        head -15 ${base_ipreg_fname}
    fi
    test -s ${base_ipreg_fname}
    
    cat ${base_ipreg_fname} \
    | ${bin_dir}/filter-eu-ranges.py --geodata ${geodata_fname} \
    | make_ipreg_compacted ${geodata_fname} \
    | ${bin_dir}/ipreg-fmt --add-json-attr is_eu \
    | ${bin_dir}/ipreg-add-stub \
    > ${merged_ipreg_fname}

    local mm_anon_ipreg_fname="${base_ipreg_fname}_mmanon"
    local mm_anon_patch_fname="${data_dir}/IPREG-mmanon.patch"

    prepare_maxmind_anon_patch ${mm_anon_patch_fname}
    cat ${merged_ipreg_fname} | apply_patch ${mm_anon_patch_fname} > ${mm_anon_ipreg_fname}

    local ipreg_final_fname="${data_dir}/IPREG.final"
    local traits_section_fname="${data_dir}/IPREG-traits.list"

    cat ${mm_anon_ipreg_fname} \
    | grep '"is_eu":1' \
    | ${bin_dir}/ipreg-fmt --remove-json-attr is_eu | ${bin_dir}/ipreg-merge | ${bin_dir}/ipreg-add-stub \
    | add_dot_reg_id_attr \
    | sort_json \
    | del_dot_reg_id_attr \
    | tee ${mm_anon_ipreg_fname}.sort \
    | prepare_traits_section \
    | tee ${traits_section_fname}.tmp \
    | column1_only \
    | traits_postproc \
    > ${traits_section_fname}

    rm ${data_dir}/GeoIP2-Anonymous-IP*.csv
    cat ${mm_anon_ipreg_fname} \
    | grep '"is_eu":1' \
    | grep -E '"vpn"|"proxy"|"hosting"|"tor"' \
    > ${data_dir}/GeoIP2-Anonymous-IP-Blocks-IP46.csv

    cat ${mm_anon_ipreg_fname}.sort \
    | ${script_dir}/replace_traits_by_idx.py --traits ${traits_section_fname}.tmp --debug-counter 250000 \
    | ${script_dir}/ipreg-fmt --split-mix-ranges \
    | ${bin_dir}/ipreg-check-addr-space --completeness \
    > ${ipreg_final_fname}
    rm ${mm_anon_ipreg_fname}.sort ${traits_section_fname}.tmp
}

main_pure_country() {
    simple_log ${FUNCNAME}
    prepare_ipreg_utils

    local geodata_fname="${data_dir}/geodata6.bin"

    if [ ! -s ${geodata_fname} ]; then
        download_file $(get_geodata_url) > ${geodata_fname}
    fi
    test -s ${geodata_fname}

    if [ ! -f ${data_dir}/add_flags.config ]; then
        echo 'IPREG_RES_URL="https://proxy.sandbox.yandex-team.ru/last/IPREG_DATA"' > ${data_dir}/add_flags.config
        echo 'IPREG_JSON_PATH="IPREG.nousr.country.json?owner=IPREG&attrs=\\{\"kind\":\"internal\",\"source\":\"ipreg\",\"name\":\"nousr\"\\}"' >> ${data_dir}/add_flags.config
    fi
    . ${data_dir}/add_flags.config # IPREG_RES_URL / IPREG_JSON_PATH

    local base_ipreg_fname="${data_dir}/IPREG.json"

    if [ ! -s ${base_ipreg_fname} ]; then
        download_file "${IPREG_RES_URL}/${IPREG_JSON_PATH}" > ${base_ipreg_fname}

        ls -al ${base_ipreg_fname}
        head -15 ${base_ipreg_fname}
    fi
    test -s ${base_ipreg_fname}

    local merged_ipreg_fname="${base_ipreg_fname}_wo_reliabilities"

    cat ${base_ipreg_fname} \
    | ${bin_dir}/ipreg-fmt --remove-json-attr reliability \
    > ${merged_ipreg_fname}
    
    local traits_section_fname="${data_dir}/IPREG-traits.list"

    cat ${base_ipreg_fname} \
    | prepare_traits_section \
    | tee ${traits_section_fname}.tmp \
    | column1_only \
    | traits_postproc \
    > ${traits_section_fname}

    local ipreg_final_fname="${data_dir}/IPREG.final"

    cat ${base_ipreg_fname} \
    | ${script_dir}/replace_traits_by_idx.py --traits ${traits_section_fname}.tmp --debug-counter 25000 \
    | ${script_dir}/ipreg-fmt --split-mix-ranges \
    | ${bin_dir}/ipreg-check-addr-space --completeness \
    > ${ipreg_final_fname}
    rm ${traits_section_fname}.tmp
}

patch_mode=$2
main_${patch_mode}
