#!/usr/bin/env python
# -*- coding: utf-8 -*-

# helpers for processing of networks with project-id

import ipaddr
import sys


def is_net_with_project_id(net_str):
    return net_str.endswith("/40") and '@' in net_str


def conv_net_with_project_id(net_str):
    if not is_net_with_project_id(net_str):
        return net_str

    net_wo_mask_len = net_str.split('/')[0]
    proj_code, net_prefix = net_wo_mask_len.split('@')

    max_proj_id_hex_len = 8
    proj_code_len = len(proj_code)
    if proj_code_len == 0 or proj_code_len > max_proj_id_hex_len:
        raise Exception('>>> bad projec-id part // %s => %s' % (net_str, proj_code))

    ipv6_parts = str(ipaddr.IPAddress(net_prefix).exploded).split(':')

    proj_id_part1 = "0000"
    proj_id_part2 = proj_code

    id_part_size = 4
    if proj_code_len > id_part_size:
        part2_idx = proj_code_len - id_part_size
        proj_id_part1 = proj_code[:part2_idx]
        proj_id_part2 = proj_code[part2_idx:]

    ipv6_parts[4] = proj_id_part1
    ipv6_parts[5] = proj_id_part2

    new_net_str = str(ipaddr.IPAddress(":".join(ipv6_parts))) + "/96"
    return new_net_str


def stream_processing(in_stream, out_stream, mark=""):
    for item in in_stream:
        net = item.strip()
        if is_net_with_project_id(net):
            tuned_net = conv_net_with_project_id(net)
            print >>sys.stderr, ">>> NB: net with proj_id detected %s -> %s\t%s" % (net, tuned_net, mark)
            net = tuned_net
        print >>out_stream, net


def check_correctness_inplace():
    test_data_pairs = [
        # equal result
          ["non-proj-id-net", "non-proj-id-net"]
        , ["proj-id@net", "proj-id@net"]
        , ["net/40", "net/40"]
        , ["1:2::/5", "1:2::/5"]
        , ["@1:2::/5", "@1:2::/5"]
        # proj-id-net detection failure
        , ["@/40", None]
        , ["@1:2:3::/40", None]
        , ["123456789@1:2:3::/40", None]
        # correct processing
        , ["6@1:2:3::/40", "1:2:3::6:0:0/96"]
        , ["6123@1:2:3::/40", "1:2:3::6123:0:0/96"]
        , ["50612@1:2:3::/40", "1:2:3:0:5:612::/96"]
        , ["51236123@1:2:3::/40", "1:2:3:0:5123:6123::/96"]
    ]

    for test_data in test_data_pairs:
        orig_net, wanted_net = test_data
        result_net = None

        try:
            result_net = conv_net_with_project_id(orig_net)
        except:
            pass

        if result_net != wanted_net:
            raise Exception(">>> internal assert: %s => %s (!= %s)" % (orig_net, wanted_net, result_net))


if __name__ == "__main__":
    check_correctness_inplace()
    stream_processing(sys.stdin, sys.stdout)


#
# NB: @see any possible project_id
# https://racktables.yandex-team.ru/index.php?page=services&tab=projects
#
