# -*- coding: utf-8 -*-

import logging
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.geosearch.tools import stat
from sandbox.projects import resource_types as rtypes
from sandbox.projects.geosearch.tools.nanny import Nanny
from sandbox.projects.geosearch import resource_types as geo_types
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.geosearch.CreateAddrsBeta import CreateAddrsBeta
from sandbox.projects.geosearch.AddrsRatingsTest import AddrsRatingsTest
from sandbox.projects.geosearch.tools.database_notifications import notify_by_telegram
from sandbox.projects.geosearch.AddrsMapkitCompareTest import AddrsMapkitCompareTest
from sandbox.projects.geosearch.AddrsBasePerformanceParallelAcceptance import AddrsBasePerformanceParallelAcceptance
from sandbox.projects.MapsDatabaseBusinessQualityAcceptance import MapsDatabaseBusinessQualityAcceptance
from sandbox.projects.geosearch.AddrsBaseAutoreleaser import AddrsBaseAutoreleaser


BASE_MLM_TEMPLATE = 'arcadia:/arc/trunk/arcadia/search/metrics_templates/geo/base.json'


class GeoAcceptanceException(Exception):
    pass


class AcceptanceGeobasesearchDatabase(sdk2.Task):
    '''
        Addrs basesearch database acceptance
    '''

    class Parameters(sdk2.task.Parameters):
        shardmap = sdk2.parameters.Resource('Shardmap to release',
                                            resource_type=geo_types.ADDRS_BASE_YP_SHARDMAP,
                                            required=True)
        shards_count = sdk2.parameters.Integer('Number of shards',
                                               default_value=4,
                                               required=True)
        geobasesearch = sdk2.parameters.Resource('Geobasesearch resource',
                                                 resource_type=rtypes.GEOBASESEARCH_EXECUTABLE)
        geobasesearch_reference = sdk2.parameters.Resource('Geobasesearch resource (reference if prod is not ok for you)',
                                                           resource_type=rtypes.GEOBASESEARCH_EXECUTABLE)
        surl_resource = sdk2.parameters.Resource('SURL resource id',
                                                    resource_type=rtypes.MAPS_WIZARD_ORG_NAV_DATA)
        wizard_data_resource = sdk2.parameters.Resource('Wizard ppo data resource id',
                                                           resource_type=rtypes.MAPS_DATABASE_BUSINESS_WIZARD_DATA)

        routes_resource = sdk2.parameters.Resource('Routes gzt for Transit resource id',
                                                   resource_type=geo_types.GEOSEARCH_TRANSIT_GZT)

        mlm_template = sdk2.parameters.String('MLM template id')
        startrek_task_id = sdk2.parameters.String('Startrek task id')
        export_real_path = sdk2.parameters.String('Path to altay export')

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    def _get_test_basesearch_service(self):
        for service in self.all_test_services:
            if 'addrs_base' in service:
                return service

    def _get_shardmap_from_prod(self, nanny):
        prod_service = 'addrs_base'
        runtime_attrs = nanny.get_sandbox_files(prod_service)
        shardmap_resource = runtime_attrs['sandbox_bsc_shard']['sandbox_shardmap']
        shardmap_task_id = shardmap_resource['task_id']
        shardmap_task = sdk2.Task[shardmap_task_id]
        shardmap = sdk2.Resource[geo_types.ADDRS_BASE_YP_SHARDMAP].find(task=shardmap_task).first()
        return shardmap.id

    def _get_resource_from_prod(self, nanny, resource_type):
        prod_service = 'addrs_base'
        runtime_attrs = nanny.get_sandbox_files(prod_service)
        sandbox_files = runtime_attrs['sandbox_files']
        for sandbox_file in sandbox_files:
            # if sandbox_file.get('resource_type') == 'GEOBASESEARCH_EXECUTABLE':
            if sandbox_file.get('resource_type') == resource_type:
                build_task_id = sandbox_file.get('task_id')
                build_task = sdk2.Task[build_task_id]
        resource = sdk2.Resource[resource_type].find(task=build_task).first()
        return resource.id

    def _get_binary_reference(self, nanny):
        if self.Parameters.geobasesearch_reference:
            return self.Parameters.geobasesearch_reference
        return self._get_resource_from_prod(nanny, 'GEOBASESEARCH_EXECUTABLE')

    def _additional_resources_warning(self):
        if not self.Parameters.surl_resource:
            surl_warn = (u'Внимание! В таск %s не передан параметр '
                         u'"SURL resource id".\n Проверьте таск варки '
                         u'базы' % self.id)
            self.startrek.add_comment(self.Parameters.startrek_task_id, surl_warn)
        if not self.Parameters.wizard_data_resource:
            wizard_data_warn = (u'Внимание! В таск %s не передан параметр '
                                u'"Wizard ppo data resource id".\n'
                                u'Проверьте таск варки базы' % self.id)
            self.startrek.add_comment(self.Parameters.startrek_task_id,
                                      wizard_data_warn)
        if not self.Parameters.routes_resource:
            routes_warn = (u'Внимание! В таск %s не передан параметр '
                           u'"Routes gzt for Transit resource id".\n'
                           u'Проверьте таск варки базы' % self.id)
            self.startrek.add_comment(self.Parameters.startrek_task_id,
                                      routes_warn)

    def _beta_warning(self):
        betas = {self.Context.create_ref_base_task_id: self.Context.ref_base_data,
                 self.Context.create_test_base_task_id: self.Context.test_base_data,
                 self.Context.create_ref_wizard_task_id: self.Context.ref_wizard_data,
                 self.Context.create_test_wizard_task_id: self.Context.test_wizard_data}
        for task, data in betas.iteritems():
            if not data:
                task_obj = sdk2.Task[task]
                if task_obj.status in self.bad_statuses:
                    message = (
                        '{acceptance_ticket} acceptance subtask {subtask_type} '
                        '(#{subtask_id}) for {beta} {beta_type} failed\n'
                        'https://sandbox.yandex-team.ru/task/{subtask_id}/view'
                    ).format(
                        acceptance_ticket=self.Parameters.startrek_task_id,
                        subtask_id=task,
                        subtask_type=task_obj.type,
                        beta=task_obj.Parameters.beta_type_choices,
                        beta_type=task_obj.Parameters.tag
                    )
                    self.notify_by_telegram('', message)
                beta_name = 'addrs_{beta_type}-{branch}-{tag}'.format(beta_type=task_obj.Context.beta_type_choices,
                                                                      branch=task_obj.Context.branch,
                                                                      tag=task_obj.Context.tag)
                comment = (
                    u'Не удалось получить данные о бете ((https://yappy.z.yandex-team.ru/b/{beta} {beta}))\n'
                    u'Проверьте таск https://sandbox.yandex-team.ru/task/{task_id}/view'
                ).format(beta=beta_name,
                         task_id=task_obj.id)
                self.startrek.add_comment(self.Parameters.startrek_task_id, comment)

    def _make_addrs_url(self, base_data, wizard_data):
        url_tpl = 'http://addrs-testing.search.yandex.net/search/stable/yandsearch?{experimental_sources}'
        experimental_sources = ['&source=%s' % source for source in base_data['experimental_sources']]
        if wizard_data:
            experimental_sources.extend(['&source=%s' % source for source in wizard_data['experimental_sources']])
        experimental_sources_string = ''.join(experimental_sources)
        return url_tpl.format(experimental_sources=experimental_sources_string)

    def launch_autoreleaser(self):
        autorelease_task_class = sdk2.Task[AddrsBaseAutoreleaser.type]
        autorelease_task = autorelease_task_class(self,
                                                  base_acceptance_task_id=self.id,
                                                  owner=self.owner,
                                                  kill_timeout=self.Parameters.kill_timeout)
        autorelease_task.enqueue()

    def notify_by_telegram(self, event='', message=None):
        if not message:
            message = ('Geosearch base {task} acceptance {event}\n'
                       'Details can be found in '
                       'https://st.yandex-team.ru/{task}').format(task=self.Parameters.startrek_task_id, event=event)
        notify_by_telegram(self, event, message)

    def on_failure(self, prev_status):
        self.notify_by_telegram('failed')

    def on_break(self, prev_status, status):
        self.notify_by_telegram('failed  with exception')

    def on_timeout(self, prev_status):
        self.notify_by_telegram('timed out')

    def on_execute(self):
        with self.memoize_stage.STAT_ACCEPTANCE_START(commit_on_entrance=False):
            stat.push_to_stat_table('sprav.yandex.ru/db/db_by_stage',
                                    self.Parameters.export_real_path,
                                    'start_geosearch_index_acceptance')
        token = sdk2.Vault.data('robot-geosearch', 'ADDRS')
        nanny = Nanny(token)
        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        self.startrek = StartrekClient(startrek_token)
        self.bad_statuses = [
            'FAILURE',
            'EXCEPTION',
            'TIMEOUT'
        ]
        self.reference_base_resources = []
        if self.Parameters.geobasesearch_reference:
            self.reference_base_resources.append(self.Parameters.geobasesearch_reference.id)
        self.Context.reference_base_resources = self.reference_base_resources
        self.reference_wizard_resources = []
        self.Context.reference_wizard_resources = self.reference_wizard_resources
        self.test_base_resources = []
        if self.Parameters.geobasesearch:
            self.test_base_resources.append(self.Parameters.geobasesearch.id)
        if self.Parameters.shardmap:
            self.test_base_resources.append(self.Parameters.shardmap.id)
        self.Context.test_base_resources = self.test_base_resources
        self.test_wizard_resources = []
        if self.Parameters.surl_resource:
            self.test_wizard_resources.append(self.Parameters.surl_resource.id)
        if self.Parameters.wizard_data_resource:
            self.test_wizard_resources.append(self.Parameters.wizard_data_resource.id)
        if self.Parameters.routes_resource:
            self.test_wizard_resources.append(self.Parameters.routes_resource.id)
        self.Context.test_wizard_resources = self.test_wizard_resources
        with self.memoize_stage.ADDITIONAL_RESOURCES_WARNING:
            self._additional_resources_warning()
        with self.memoize_stage.PREPARE_BETAS(commit_on_entrance=False):
            create_beta_class = sdk2.Task[CreateAddrsBeta.type]
            create_ref_base = create_beta_class(create_beta_class.current,
                                                branch=self.Parameters.startrek_task_id.lower(),
                                                tag='reference',
                                                resources=self.reference_base_resources,
                                                beta_type_choices='base',
                                                startrek_task=self.Parameters.startrek_task_id,
                                                launch_type_choices='DB',
                                                owner=self.owner,
                                                create_sub_task=True).enqueue()
            create_test_base = create_beta_class(create_beta_class.current,
                                                 branch=self.Parameters.startrek_task_id.lower(),
                                                 tag='test',
                                                 resources=self.test_base_resources,
                                                 beta_type_choices='base',
                                                 startrek_task=self.Parameters.startrek_task_id,
                                                 launch_type_choices='DB',
                                                 owner=self.owner,
                                                 create_sub_task=True).enqueue()
            create_ref_wizard = create_beta_class(create_beta_class.current,
                                                  branch=self.Parameters.startrek_task_id.lower(),
                                                  tag='reference',
                                                  resources=self.reference_wizard_resources,
                                                  beta_type_choices='wizard',
                                                  startrek_task=self.Parameters.startrek_task_id,
                                                  launch_type_choices='DB',
                                                  owner=self.owner,
                                                  create_sub_task=True).enqueue()
            create_test_wizard = create_beta_class(create_beta_class.current,
                                                   branch=self.Parameters.startrek_task_id.lower(),
                                                   tag='test',
                                                   resources=self.test_wizard_resources,
                                                   beta_type_choices='wizard',
                                                   startrek_task=self.Parameters.startrek_task_id,
                                                   launch_type_choices='DB',
                                                   owner=self.owner,
                                                   create_sub_task=True).enqueue()
            self.Context.create_ref_base_task_id = create_ref_base.id
            self.Context.create_test_base_task_id = create_test_base.id
            self.Context.create_ref_wizard_task_id = create_ref_wizard.id
            self.Context.create_test_wizard_task_id = create_test_wizard.id
            raise sdk2.WaitTask([self.Context.create_ref_base_task_id,
                                 self.Context.create_test_base_task_id,
                                 self.Context.create_ref_wizard_task_id,
                                 self.Context.create_test_wizard_task_id],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)
        self.Context.ref_base_data = sdk2.Task[self.Context.create_ref_base_task_id].Context.chaining_dict
        self.Context.test_base_data = sdk2.Task[self.Context.create_test_base_task_id].Context.chaining_dict
        self.Context.ref_wizard_data = sdk2.Task[self.Context.create_ref_wizard_task_id].Context.chaining_dict
        self.Context.test_wizard_data = sdk2.Task[self.Context.create_test_wizard_task_id].Context.chaining_dict
        with self.memoize_stage.BETA_WARNING(commit_on_entrance=False):
            self._beta_warning()
        with self.memoize_stage.ACCEPTANCE(commit_on_entrance=False):
            subtasks = []
            if self.Parameters.geobasesearch:
                test_binary = self.Parameters.geobasesearch.id
            else:
                test_binary = self._get_resource_from_prod(nanny, 'GEOBASESEARCH_EXECUTABLE')

            performance_task = AddrsBasePerformanceParallelAcceptance(
                self,
                dolbilka_executor_max_simultaneous_requests=50,
                dolbilka_executor_mode='finger',
                dolbilka_executor_requests_limit=500000,
                dolbilka_executor_sessions=5,
                dolbilka_request_timeout=600,
                dolbilo_plan_resource_id=sdk2.Resource[rtypes.BASESEARCH_PLAN].find(attrs={'type': 'addrs_business'}).first().id,
                fast_features_sharded_resource_id=sdk2.Resource[geo_types.MAPS_DATABASE_FAST_FEATURES_SHARD].find(attrs={'released': 'stable'}).first().id,
                geobasesearch_config_resource_id=sdk2.Resource[rtypes.MAPS_SEARCH_CONFIG_BASE].find(attrs={'released': 'stable'}).first().id,
                maps_search_advert_db_resource_id=sdk2.Resource[rtypes.MAPS_DATABASE_ADVERT].find(attrs={'released': 'stable'}).first().id,
                # profiling_type='none',  # WARNING - any profiling type incompatitble with memory measurements
                binary1=str(self._get_binary_reference(nanny)),
                shardmap1=str(self._get_shardmap_from_prod(nanny)),
                launch_type='DB',
                startrek_task=self.Parameters.startrek_task_id,
                push_data=True,
                binary2=test_binary,
                shardmap2=self.Parameters.shardmap.id,
                offline_warmup_time=1,
                offline_shutdown_time=1,
                filter_stat_resource_id=sdk2.Resource[rtypes.BUSINESS_COMPUTED_FILTERS].find(attrs={'released': 'stable'}).first().id,
                owner=self.owner,
                create_sub_task=True,
                kill_timeout=self.Parameters.kill_timeout).enqueue()
            self.Context._PERFORMANCE_ACCEPTANCE_KEY = performance_task.id
            subtasks.append(performance_task.id)
            with self.memoize_stage.PERFORMANSE_TASK_INFO:
                self.startrek.add_comment(self.Parameters.startrek_task_id,
                                          ('Performance acceptance task\n'
                                           'Task: https://sandbox.yandex-team.ru/task/%s/view') % performance_task.id)
            quality_acceptance_task_class = sdk2.Task[MapsDatabaseBusinessQualityAcceptance.type]
            logging.info('Trying to start MLM acceptance')
            logging.info('Additional resources %s' % self.Context.test_wizard_resources)
            with self.memoize_stage.MLM_QUALITY_ACCEPTANCE(commit_on_entrance=False):
                production = self.Context.ref_base_data['experimental_sources']
                experimental = self.Context.test_base_data['experimental_sources']
                if self.Context.test_wizard_resources and self.Context.test_wizard_data and self.Context.ref_wizard_data:
                    production.extend(self.Context.ref_wizard_data['experimental_sources'])
                    experimental.extend(self.Context.test_wizard_data['experimental_sources'])
                mlm_acceptance_task = quality_acceptance_task_class(quality_acceptance_task_class.current,
                                                                          mlm_template_path=BASE_MLM_TEMPLATE,
                                                                          startrek_task_id=self.Parameters.startrek_task_id,
                                                                          vault_key_name='robot-geosearch-base',
                                                                          mark='MLM',
                                                                          production_sources=production,
                                                                          experimental_sources=experimental,
                                                                          owner=self.owner,
                                                                          create_sub_task=True,
                                                                          kill_timeout=self.Parameters.kill_timeout).enqueue()
                self.Context._MLM_ACCEPTANCE_KEY = mlm_acceptance_task.id
                subtasks.append(mlm_acceptance_task.id)
            if self.Context.ref_wizard_data and self.Context.test_wizard_data:
                reference_url = self._make_addrs_url(self.Context.ref_base_data,
                                                     self.Context.ref_wizard_data)
                test_url = self._make_addrs_url(self.Context.test_base_data,
                                                self.Context.test_wizard_data)
            else:
                reference_url = self._make_addrs_url(self.Context.ref_base_data, {})
                test_url = self._make_addrs_url(self.Context.test_base_data, {})
            logging.info('Reference URL: {url}'.format(url=reference_url))
            logging.info('Test URL: {url}'.format(url=test_url))
            mapkit_acceptance_task_class = sdk2.Task[AddrsMapkitCompareTest.type]
            mapkit_acceptance_task = mapkit_acceptance_task_class(mapkit_acceptance_task_class.current,
                                                                  reference_stand_url=reference_url,
                                                                  test_stand_url=test_url,
                                                                  launch_type_choices='DB',
                                                                  startrek_task=self.Parameters.startrek_task_id,
                                                                  kill_timeout=self.Parameters.kill_timeout,
                                                                  owner=self.owner,
                                                                  create_sub_task=True).enqueue()
            self.Context._MAPKIT_ACCEPTANCE_KEY = mapkit_acceptance_task.id
            subtasks.append(mapkit_acceptance_task.id)
            self.launch_autoreleaser()
            ratings_test_task_class = sdk2.Task[AddrsRatingsTest.type]
            ratings_test_task = ratings_test_task_class(ratings_test_task_class.current,
                                                        reference_stand_url=reference_url,
                                                        test_stand_url=test_url,
                                                        startrek_task=self.Parameters.startrek_task_id,
                                                        kill_timeout=self.Parameters.kill_timeout,
                                                        owner=self.owner,
                                                        create_sub_task=True).enqueue()
            self.Context._RATINGS_TEST_KEY = ratings_test_task.id
            raise sdk2.WaitTask(subtasks,
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)
            for task_id in subtasks:
                task_obj = sdk2.Task[task_id]
                if task_obj.status in self.bad_statuses:
                    message = (
                        '{acceptance_ticket} acceptance subtask {subtask_type} '
                        '(#{subtask_id}) failed\n'
                        'https://sandbox.yandex-team.ru/task/{subtask_id}/view'
                    ).format(
                        acceptance_ticket=self.Parameters.startrek_task_id,
                        subtask_id=task_id,
                        subtask_type=task_obj.type
                    )
                    self.notify_by_telegram('', message)
        if not self.Context.finish_time_sent:
            with self.memoize_stage.STAT_ACCEPTANCE_FINISH(commit_on_entrance=False):
                stat.push_to_stat_table('sprav.yandex.ru/db/db_by_stage',
                                        self.Parameters.export_real_path,
                                        'finish_geosearch_index_acceptance')
            self.Context.finish_time_sent = True
