# -*- coding: utf-8 -*-

import os
import time
import pickle
import jinja2
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
import sandbox.common.types.notification as ctn
from sandbox.common.errors import UnknownTaskType
from sandbox.projects.geosearch.tools.misc import retry
from sandbox.projects.geosearch.tools.nanny import Nanny
from sandbox.projects.geosearch import resource_types as rtypes
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.geosearch.tools.database_notifications import who_is_on_duty
from sandbox.projects.geosearch.AddrsBaseAutoreleaser import utils

from metric import LaunchSet


SLEEP_BEFORE_RELEASE = 15 * 60


class AddrsBaseAutoreleaser(sdk2.Task):
    """ Check if geosearch database is ready for release"""

    class Parameters(sdk2.task.Parameters):
        base_acceptance_task_id = sdk2.parameters.Integer('ACCEPTANCE_GEOBASESEARCH_DATABASE task ID')

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2048
        environments = (
            environments.PipEnvironment('pytz', use_wheel=True),
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_acceptance_task(self):
        return sdk2.Task[self.Parameters.base_acceptance_task_id]

    def get_mlm_launch_ids(self):
        acceptance_task = self.get_acceptance_task()
        try:
            mlm_task = sdk2.Task[acceptance_task.Context._MLM_ACCEPTANCE_KEY]
        except UnknownTaskType:
            logging.info('Waiting for MAPS_DATABASE_BUSINESS_QUALITY_ACCEPTANCE tasks to start')
            raise sdk2.WaitTime(180)
        if 'mlm_task_id' not in mlm_task.Context:
            logging.info('Waiting for MLM launch to start')
            raise sdk2.WaitTime(180)
        return {
            mlm_task.Parameters.mark: mlm_task.Context.mlm_task_id,
        }

    def get_perf_task(self):
        acceptance_task = self.get_acceptance_task()
        return sdk2.Task[acceptance_task.Context._PERFORMANCE_ACCEPTANCE_KEY]

    def get_marker_test_task(self):
        acceptance_task = self.get_acceptance_task()
        return sdk2.Task[acceptance_task.Context._MAPKIT_ACCEPTANCE_KEY]

    @retry(tries=10, delay=120)
    def check_marker_test(self):
        task = self.get_marker_test_task()
        logging.info('{task_type} status is {task_status}'.format(task_type=task.type,
                                                                  task_status=task.status))
        if task.status == 'SUCCESS':
            self.Context.MARKER_TESTS_DIFF = task.Context.diff

    def check_perf_tests(self):
        task = self.get_perf_task()
        self.Context.PERF_TEST = {}
        logging.info('{task_type} status is {task_status}'.format(task_type=task.type,
                                                                  task_status=task.status))
        thresholds = {
            'rss': 3.0,
            'vsz': 3.0,
            'latency_95': 10.0,
            'latency_99': 15.0,
        }
        if task.status == 'SUCCESS':
            for metric in task.Context.result:
                name = metric[0]
                values = metric[1]
                if name != 'rps':
                    if round(values.get('percent', 0.0)) > thresholds.get(name):
                        self.Context.PERF_TEST.update({name: values})

    def completed(self):
        statuses = set()
        for launch_set in self.launch_sets:
            statuses.add(launch_set.status)
            logging.info('{} status is {}'.format(launch_set.name, launch_set.status))
        logging.info('MLM statuses is {}'.format(statuses))
        return len(statuses) == 1 and statuses.pop() == 'READY'

    def autodeploy(self):
        if not self.Context.PERF_TEST:
            for launch_set in self.launch_sets:
                logging.info('{} ready for release: {}'.format(launch_set.name, launch_set.release))
                if launch_set.release:
                    return launch_set.name

    def check_all(self):
        token = sdk2.Vault.data('GEOSEARCH_PROD', 'robot-geosearch-base')
        while True:
            self.launch_sets = []
            for name, launch_set_id in self.Context.mlm_launch_ids.iteritems():
                self.launch_sets.append(LaunchSet(name, launch_set_id, token))
            self.check_marker_test()
            self.check_perf_tests()
            if self.completed():
                logging.info('MLM launches is completed')
                break
            else:
                raise sdk2.WaitTime(600)

    def make_startrek_message(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        comment = env.get_template("startrek.tpl").render(self.Context.data_to_render)
        if self.Context.AUTODEPLOY:
            return '===!!(green)Можно катить базу и визардные данные!!\n{comment}'.format(comment=comment)
        return '===!!(red)Не рекомендуется выкатывать базу в продакшн!!\n{comment}'.format(comment=comment)

    def write_to_startrek(self):
        acceptance_task = self.get_acceptance_task()
        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        self.startrek = StartrekClient(startrek_token)
        self.startrek.add_comment(acceptance_task.Parameters.startrek_task_id,
                                  self.make_startrek_message())
        if self.autodeploy():
            tag = 'autoacceptance_ok'
        else:
            tag = 'autoacceptance_failed'
        self.startrek.add_tag(acceptance_task.Parameters.startrek_task_id, tag)

    def pickle_launch_sets(self):
        for launch_set in self.launch_sets:
            try:
                filename = './{}.dump'.format(launch_set.name)
                with open(filename, 'w') as dump_file:
                    pickle.dump(launch_set, dump_file)
                autorelease_dump_resource = sdk2.Resource[rtypes.ADDRS_BASE_AUTORELEASE_DUMP]
                current_autorelease_dump_resource = autorelease_dump_resource(self,
                                                                              '{} dump'.format(launch_set.name),
                                                                              filename)
                autorelease_dump = sdk2.ResourceData(current_autorelease_dump_resource)
                autorelease_dump.ready()
            except Exception as err:
                logging.info('Failed to pickle lanuch set {}: {}'.format(launch_set, err))

    def check_release_time(self):
        mourning_hour = 10
        hours_to_sleep = utils.get_sleep_to_release_hours(morning=mourning_hour)
        if hours_to_sleep:
            logging.info('Sleeping until {} am'.format(mourning_hour))
            raise sdk2.WaitTime(hours_to_sleep * 60 * 60)

    def _get_builder_shardmap_resource(self):
        acceptance_task = self.get_acceptance_task()
        return sdk2.Resource[acceptance_task.Parameters.shardmap]

    def _get_acceptance_ticket(self):
        acceptance_task = self.get_acceptance_task()
        return acceptance_task.Parameters.startrek_task_id

    def _get_base_build_task(self):
        builded_shardmap_resource = self._get_builder_shardmap_resource()
        base_build_task = sdk2.Task[builded_shardmap_resource.task_id]
        return base_build_task

    def newer_base_released(self):
        nanny_cli = Nanny(sdk2.Vault.data('robot-geosearch', 'ADDRS'))
        base_build_task = self._get_base_build_task()
        production_resources = nanny_cli.get_sandbox_files('addrs_base')
        production_shardmap = production_resources['sandbox_bsc_shard']['sandbox_shardmap']
        production_task_id = int(production_shardmap.get('task_id'))
        if production_task_id > base_build_task.id:
            return 'https://sandbox.yandex-team.ru/task/{}/view'.format(production_task_id)
        return False

    def newer_base_accepted(self):
        autorelease_tasks = sdk2.Task.find(self.type,
                                           status=tuple(ctt.Status.Group.EXECUTE) + tuple(ctt.Status.Group.WAIT)).order(-sdk2.Task.id).limit(10)
        latest_autorelease_task = autorelease_tasks.first()
        logging.info('latest autorelease task: {}'.format(latest_autorelease_task))
        if latest_autorelease_task:
            if latest_autorelease_task.id > self.id and latest_autorelease_task.Context.AUTODEPOY:
                latest_acceptance_task = sdk2.Task[latest_autorelease_task.base_acceptance_task_id]
                return 'https://st.yandex-team.ru/{}'.format(latest_acceptance_task.Context.startrek_task_id)
        return False

    def newer_base_available(self):
        return self.newer_base_accepted() or self.newer_base_released()

    def release(self):
        on_duty = who_is_on_duty()
        on_duty.append('karas-pv')
        base_build_task = self._get_base_build_task()
        autodeploy = self.autodeploy()
        if autodeploy:
            self.check_release_time()
            message = ('Geosearch base {ticket} is about to release as {autodeploy} '
                       'in {sleep_time} minutes.\nIf you want to cancel '
                       'automatic release for some reason, stop task {autorelease_task} '
                       'https://sandbox.yandex-team.ru/task/{autorelease_task}/view '
                       'https://st.yandex-team.ru/{ticket}').format(ticket=self.Context.acceptance_ticket,
                                                                    autodeploy=autodeploy,
                                                                    sleep_time=SLEEP_BEFORE_RELEASE / 60,
                                                                    autorelease_task=self.id)
            self.server.notification(body=message,
                                     recipients=on_duty,
                                     transport=ctn.Transport.TELEGRAM)
            logging.info('Telegram notification on release sended to {on_duty}'.format(on_duty=on_duty))
            logging.info('Sleeping for {sleep_time} seconds'.format(sleep_time=SLEEP_BEFORE_RELEASE))
            time.sleep(SLEEP_BEFORE_RELEASE)
            newer_base = self.newer_base_available()
            if newer_base:
                message = ('{ticket} automatic release canceled because newer '
                           'task {newer_base} released or will be released '
                           'soon').format(ticket=self.Context.acceptance_ticket,
                                          newer_base=newer_base)
                self.server.notification(body=message,
                                         recipients=on_duty,
                                         transport=ctn.Transport.TELEGRAM)
            else:
                self.server.release(
                    params={'task_id': base_build_task.id,
                            'message': autodeploy,
                            'type': 'stable',
                            'params': {},
                            'subject': 'Regular geosearch base release'}
                )
        else:
            message = ('Geosearch base {ticket} didn`t pass autoreleaser checks. Details could be found in '
                       'acceptance ticket: https://st.yandex-team.ru/{ticket}\n'
                       '{ticket} still can be released manually.\n'
                       'To release the base without wizard data, add "PURE_BASE" to the release comment.\n'
                       'https://sandbox.yandex-team.ru/task/{build_task}/view').format(ticket=self.Context.acceptance_ticket,
                                                                                       build_task=base_build_task.id)
            self.server.notification(body=message,
                                     recipients=on_duty,
                                     transport=ctn.Transport.TELEGRAM)
            logging.info('Telegram notification sended to {on_duty}'.format(on_duty=on_duty))

    def on_execute(self):
        self.Context.mlm_launch_ids = self.get_mlm_launch_ids()
        self.check_all()
        self.Context.acceptance_ticket = self._get_acceptance_ticket()
        self.Context.data_to_render = {
            'launch_sets': [launch_set.text() for launch_set in self.launch_sets],
            'marker_data': self.Context.MARKER_TESTS_DIFF,
            'perf_data': self.Context.PERF_TEST
        }
        self.Context.AUTODEPLOY = self.autodeploy()
        self.pickle_launch_sets()
        with self.memoize_stage.startrek_message(commit_on_entrance=False):
            self.write_to_startrek()
        self.release()
