# -*- coding: utf-8 -*-
import os
import jinja2
import requests

MLM_URL = 'https://mlm.yandex-team.ru/api/launch-set/'

BAD_SIGNIFICATIONS = {
    'common': [
    ],
    'ignore': [
    ],
    'important': [
        'RED',
        'GREEN',
    ],
    'vital': [
        'RED',
        'GREEN',
        'LIGHT_RED',
    ],
}

DIFF_THRESHOLDS = {
    'vital': {
        'LIGHT_GREEN': 0.2,
        'LIGHT_RED': -0.1,
    },
    'important': {
        'LIGHT_GREEN': 0.99,
        'LIGHT_RED': -0.15,
    },
}

METRICS = {
    # vital
    'vital': {
        'gudini': ['org1', 'rubric', 'chain', 'toponym', 'global', 'hotels'],
        'gudini_v2': ['org1', 'rubric', 'chain', 'toponym', 'global'],
        'geo-vital1_nr': ['org1', 'toponym', 'hotels'],
        'empty-serp': ['org1', 'rubric', 'chain', 'toponym', 'global', 'hotels'],
        'zoomout_kpi': ['org1', 'rubric', 'chain', 'toponym', 'global'],
    },
    # important
    'important': {
        'geo-vital5_nr': ['org1', 'toponym', 'hotels'],
        'geo-irrel-10_all': ['org1', 'rubric', 'chain', 'toponym', 'global', 'hotels'],
        'geo-p1_nr': ['org1', 'rubric', 'chain', 'toponym', 'global', 'hotels'],
        'geo-p5_nr': ['org1', 'rubric', 'chain', 'global', 'hotels'],
        'docs-found': ['org1', 'rubric', 'chain', 'toponym', 'global', 'drugs', 'hotels'],
    },
    # common
    # ignore
    'ignore': {
        'geo_click_proba_dcg10': ['org1', 'rubric', 'chain', 'toponym', 'global', 'hotels'],
        'judged1-geo_relevance_nr': [],
        'judged5-geo_relevance_nr': [],
        'judged10-geo_relevance_nr': [],
        'judged-sprav-dup-serps-10': [],
        'sprav-dup-serps-10': [],
        'zoomout_kpi': ['hotels'],
        'geo_dbd_coverage10': [],
    }
}


def _get_basket_label(basket_name):
    if 'org1' in basket_name:
        return 'org1'
    if 'rubric' in basket_name:
        return 'rubric'
    if 'chain' in basket_name:
        return 'chain'
    if 'toponym' in basket_name:
        return 'toponym'
    if 'Drugs' in basket_name:
        return 'drugs'
    if 'hotels' in basket_name or 'Hotels' in basket_name:
        return 'hotels'
    if 'duplicates' in basket_name:
        return 'duplicates'
    return 'global'


def _get_basket_category(basket_name):
    if 'labels-exussr_small' in basket_name:
        return 'important'
    if 'hotels' in basket_name or 'Hotels' in basket_name:
        return 'important'
    if 'Drugs' in basket_name:
        return 'important'
    if 'duplicates' in basket_name:
        return 'ignore'
    return 'vital'


def _categorize_metrics_basket(metric_name, basket_name):
    basket_label = _get_basket_label(basket_name)
    metric_cat = 'common'
    for category, data in METRICS.iteritems():
        for cfg_metric_name, cfg_basket_labels in data.iteritems():
            if metric_name == cfg_metric_name:
                if basket_label in cfg_basket_labels or cfg_basket_labels == []:
                    metric_cat = category
    basket_cat = _get_basket_category(basket_name)
    if basket_cat == 'ignore':
        metric_cat = 'ignore'
    if basket_cat != 'vital' and metric_cat == 'vital':
        metric_cat = 'important'
    return metric_cat


CHANGED_METRICS_CRITICAL_PERC = 4.0


BAD_VALUES = [-2, -1]

template_path = os.path.dirname(os.path.abspath(__file__))
ENV = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))


class NoThresholdException(Exception):
    pass


class MLMEntity(object):

    def __init__(self, name, data):
        self.name = name.replace('(onlySearchResult)', '')
        self.mlm_status = data.get('status')
        self.completed = self.mlm_status == 'COMLETED'
        self.failed = self.mlm_status == 'FAILED'
        self._text_template = ''

    def text(self):
        return ENV.get_template(self._text_template).render(self.dump())


class LaunchSet(MLMEntity):

    def __init__(self, name, launch_set_id, token):
        self.name = name
        self.url = MLM_URL + launch_set_id
        self.token = token
        data = self.get_data()
        self.launches = self.get_launches(data)
        self.mlm_status = data.get('status')
        self._text_template = 'launch_set.tpl'
        del(data)
        self.status = self.get_status()
        self.get_counter()
        self.changed_perc = round((self.bad_count + self.colored_count) / (self.metric_count / 100.0), 1)
        self.release = not self.bad_count and self.changed_perc < CHANGED_METRICS_CRITICAL_PERC

    def get_data(self):
        headers = {'Authorization': 'OAuth {token}'.format(token=self.token)}
        response = requests.get(self.url, headers=headers)
        return response.json()

    def get_launches(self, data):
        launches = []
        for launch in data.get('launches'):
            launch_name = launch.get('name')
            launches.append(Launch(launch_name, launch))
        return launches

    def dump(self):
        return {
            'name': self.name,
            'url': self.url,
            'launches': [launch.dump() for launch in self.launches],
            'metric_count': self.metric_count,
            'good_count': self.good_count,
            'bad_count': self.bad_count,
            'colored_count': self.colored_count,
            'changed_perc': self.changed_perc,
            'changed_critical_perc': CHANGED_METRICS_CRITICAL_PERC,
            'release': self.release,
        }

    def get_counter(self):
        self.metric_count = sum([launch.metric_count for launch in self.launches])
        self.good_count = sum([launch.good_count for launch in self.launches])
        self.bad_count = sum([launch.bad_count for launch in self.launches])
        self.colored_count = sum([launch.colored_count for launch in self.launches])

    def get_status(self):
        if self.mlm_status != 'COMPLETED':
            launches_statuses = set(filter(bool, [launch.status for launch in self.launches]))
            if len(launches_statuses) == 1:
                return launches_statuses.pop()
            else:
                return 'RUNNING'
        return 'READY'


class Launch(MLMEntity):

    def __init__(self, name, data):
        super(Launch, self).__init__(name, data)
        self.baskets = self.get_baskets(data)
        self.status = self.get_status()
        self.get_counter()
        self._text_template = 'launch.tpl'

    def get_status(self):
        if self.mlm_status != 'COMPLETED':
            baskets_statuses = set(filter(bool, [basket.status for basket in self.baskets if not basket.ignore]))
            if len(baskets_statuses) == 1:
                return baskets_statuses.pop()
            else:
                return 'RUNNING'
        return 'READY'

    def get_baskets(self, data):
        baskets = []
        for basket in data.get('diffQueryGroups'):
            if basket.get('filterName') not in ['(onlySearchResult)', '(catalogOnly)']:
                basket_name = basket.get('name') + ', ' + basket.get('filterName')
            else:
                basket_name = basket.get('name')
            baskets.append(Basket(basket_name, basket))
        return baskets

    def dump(self):
        return {
            'name': self.name,
            'baskets': filter(bool, [basket.dump() for basket in self.baskets if not basket.ignore]),
            'metric_count': self.metric_count,
            'good_count': self.good_count,
            'bad_count': self.bad_count,
            'colored_count': self.colored_count,
        }

    def get_counter(self):
        self.metric_count = sum([basket.metric_count for basket in self.baskets])
        self.good_count = sum([basket.good_count for basket in self.baskets if not basket.ignore])
        self.bad_count = sum([basket.bad_count for basket in self.baskets if not basket.ignore])
        self.colored_count = sum([basket.colored_count for basket in self.baskets if not basket.ignore])


class Basket(MLMEntity):

    def __init__(self, name, data):
        super(Basket, self).__init__(name, data)
        self.metrics = self.get_metrics(data)
        self.good_metrics = [metric for metric in self.metrics if metric.good()]
        self.bad_metrics = [metric for metric in self.metrics if metric.bad()]
        self.colored_metrics = [metric for metric in self.metrics if metric.colored()]
        self.metric_count = len(self.metrics)
        self.good_count = len(self.good_metrics)
        self.bad_count = len(self.bad_metrics)
        self.colored_count = len(self.colored_metrics)
        self.category = self.categorize()
        self.ignore = self.category == 'ignore'
        self.status = self.get_status()
        self._text_template = 'basket.tpl'

    def get_metrics(self, data):
        metrics = []
        for metric in data.get('metrics'):
            metrics.append(Metric(metric.get('metricId'), self.name, metric))
        return metrics

    def categorize(self):
        return _get_basket_category(self.name)

    def get_status(self):
        if not self.ignore:
            if self.mlm_status != 'COMPLETED':
                if not self.bad_count:
                    if not self.colored_count:
                        return 'READY'
                    else:
                        return 'RUNNING'
                if self.mlm_status == 'FAILED':
                    return 'FAILED'
                return 'RUNNING'
            return 'READY'

    def dump(self):
        if not self.ignore:
            return {
                'name': self.name,
                'bad_metrics': filter(bool, [metric.dump() for metric in self.metrics if metric.bad()]),
                'colored_metrics': filter(bool, [metric.dump() for metric in self.metrics if metric.colored()]),
                'metric_count': self.metric_count,
                'good_count': self.good_count,
                'bad_count': self.bad_count,
                'colored_count': self.colored_count,
            }


class Metric(object):

    def __init__(self, name, basket, data):
        self.name = name
        self.basket_name = basket
        self.signification = data.get('signification') or 'RED'
        self.value = data.get('value')
        self.diff_value = data.get('diffValue')
        self.diff_percent = data.get('diffPercent')
        self.baseline_value = data.get('baselineValue')
        self.category = self.categorize()
        self.ignore = self.category == 'ignore'
        # TODO: Explaination text shown on hover

    def categorize(self):
        return _categorize_metrics_basket(self.name, self.basket_name)

    def check_thresholds(self):
        thresholds = DIFF_THRESHOLDS.get(self.category)
        threshold = thresholds.get(self.signification)
        if threshold:
            if threshold < 0:
                return threshold > self.diff_percent
            if threshold > 0:
                return threshold < self.diff_percent
        else:
            raise NoThresholdException('Not found')

    def bad(self):
        if not self.ignore and self.category not in ['ignore', 'common']:
            if self.baseline_value in BAD_VALUES or self.value in BAD_VALUES:
                return True
            try:
                return self.check_thresholds()
            except NoThresholdException:
                return self.signification in BAD_SIGNIFICATIONS.get(self.category)
        return False

    def good(self):
        if not self.ignore:
            return not self.bad()
        return False

    def colored(self):
        if not self.ignore:
            return self.good() and self.signification != 'GRAY'
        return False

    def dump(self):
        if not self.ignore:
            return {
                'name': self.name,
                'signification': self.signification,
                'diff_percent': self.diff_percent,
                'category': self.category,
            }

    def _print_diff_perc(self):
        color = 'pink' if self.signification == 'LIGHT_RED' else self.signification.replace('_', '').lower()
        if not self.diff_percent or self.diff_percent == 'NaN':
            value = self.diff_percent
        else:
            value = '{:.2f}'.format(self.diff_percent)
        return '<#<a style="color: {color}; font-weight: bold;">{value}</a>#>'.format(color=color, value=value)
