# -*- coding: utf-8 -*

import os
import logging
import tarfile
import jinja2
from collections import OrderedDict

from sandbox import sdk2
from sandbox.common.share import skynet_get
import sandbox.common.types.client as ctc
import sandbox.projects.resource_types as rt
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.projects.common.geosearch.utils_sdk2 import make_parameters_cls, create_parameters_group, chop_group_suffix
from sandbox.projects.common.geosearch.search_components import MapsSearchBusiness, DEFAULT_MAPS_SEARCH_BUSINESS_PORT
from sandbox.projects.common.geosearch.search_components_sdk2 import get_addrs_basesearch_params
from sandbox.projects.common import dolbilka2
from sandbox.projects.tank import executor2 as tank_executor
from sandbox.projects.tank.load_resources import resources as tr
from sandbox.projects.tank.load_resources.resources import YANDEX_TANK_VIRTUALENV_19


class AddrsBasesearchPerformanceParallel(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 100 << 10
        ram = 50 << 10
        client_tags = ctc.Tag.INTEL_E5_2650 & ctc.Tag.LINUX_PRECISE

    base_params1 = get_addrs_basesearch_params('1')
    base_params2 = get_addrs_basesearch_params('2')
    group1 = create_parameters_group(base_params1, 'Basesearch parameters 1')
    group2 = create_parameters_group(base_params2, 'Basesearch parameters 2')

    Parameters = make_parameters_cls(OrderedDict(
        [
            ('dolbilo_plan_resource_id', sdk2.parameters.Resource('Plan', resource_type=rt.BASESEARCH_PLAN)),
            ('dolbilka_group', sdk2.parameters.Group('Dolbilka parameters')),  # empty group for separating parameters.
            ('dolbilka_param', dolbilka2.DolbilkaExecutor2.Parameters),
            ('lunapark_param', tank_executor.LunaparkPlugin.Parameters),
            ('offline_param', tank_executor.OfflinePlugin.Parameters)
        ] +
        [('addrs_group1', group1)] + list(base_params1.iteritems()) +
        [('addrs_group2', group2)] + list(base_params2.iteritems())
    ))

    class Context(sdk2.Task.Context):
        stats = {}

    @sdk2.footer()
    def footer(self):
        if '1' not in self.Context.stats or '2' not in self.Context.stats:
            return 'Calculating...'
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        return env.get_template('footer.html').render(
            run1=self.Context.stats['1'],
            run2=self.Context.stats['2']
        )

    def get_fast_features_path(self, resource_path):
        self.logger.info('All shards resource path: ' + resource_path)
        result = None
        for root, dirs, files in os.walk(resource_path):
            for file in files:
                if '.mms' in file:
                    return os.path.abspath(root)
                result = os.path.abspath(root)
        return result

    def get_addrs_basesearch(self, params, id=None, port=None):
        work_dir = self.path() if not id else self.path(str(id))
        work_dir.mkdir(parents=True, exist_ok=True)
        work_dir = str(work_dir)
        geobasesearch_params = {
            'binary': str(sdk2.ResourceData(params['geobasesearch_executable_resource_id']).path),
            'config_file': str(sdk2.ResourceData(params['geobasesearch_config_resource_id']).path)
        }
        self.logger.info('binary: %s' % geobasesearch_params['binary'])
        skynet_get(params['shard_rbtorrent'], work_dir)
        business_database_dir = os.path.join(work_dir, 'index_0000000000_0000000000')
        self.logger.info('business_database_dir: ' + business_database_dir)
        advert_database_path = None
        if params['maps_search_advert_db_resource_id'] is not None:
            advert_database_path = str(sdk2.ResourceData(params['maps_search_advert_db_resource_id']).path)
        fast_features_path = None
        if params['fast_features_all_shards_resource_id'] is not None:
            fast_features_path = self.get_fast_features_path(str(sdk2.ResourceData(params['fast_features_all_shards_resource_id']).path))
            self.logger.info('fast_features_path (all_shards): ' + fast_features_path)
        if params['fast_features_sharded_resource_id'] is not None:
            fast_features_path = str(sdk2.ResourceData(params['fast_features_sharded_resource_id']).path)
            self.logger.info('fast_features_path: ' + fast_features_path)
        filter_stat_path = None
        if params['filter_stat_resource_id'] is not None:
            filter_stat_path = str(sdk2.ResourceData(params['filter_stat_resource_id']).path)
        if port is None:
            port = DEFAULT_MAPS_SEARCH_BUSINESS_PORT + int(id)

        return MapsSearchBusiness(
            work_dir=work_dir,
            geobasesearch_params=geobasesearch_params,
            business_database_dir=business_database_dir,
            advert_database_path=advert_database_path,
            fast_features_path=fast_features_path,
            filter_stat_path=filter_stat_path,
            port=port)

    def init_virtualenv(self):
        virtualenv_resource = sdk2.Resource.find(
            type=YANDEX_TANK_VIRTUALENV_19,
            attrs={'released': 'stable'}
        ).order(-sdk2.Resource.id).first()
        virtualenv_archive_path = str(sdk2.ResourceData(virtualenv_resource).path)
        virtualenv_path = str(self.path('tank-venv'))
        paths.make_folder(virtualenv_path, delete_content=True)
        tarfile.open(virtualenv_archive_path, "r:*").extractall(virtualenv_path)
        self.logger.info('virtualenv_path: %s' % virtualenv_path)
        return virtualenv_path

    def _shoot(self, executor, name, session):
        description = "{} session {}".format(name, session)
        stats_dir = "fire.{}.{}".format(name, session)
        stats_resource = sdk2.Resource[tr.YANDEX_TANK_LOGS](self, description, stats_dir)
        stats_resource_data = sdk2.ResourceData(stats_resource)
        stats_resource_data.path.mkdir(0o755, parents=True, exist_ok=True)
        work_dir = str(stats_resource_data.path)
        self.logger.info('shoot work_dir: %s' % work_dir)

        # Note: cgroup is too restrictive to obtain maximum possible rps
        artifacts_dir = executor.fire(
            work_dir,
            job_name=description,
            cgroup=None,
            autostop_expected=False,
        )
        return stats_resource, artifacts_dir

    def _memory_usage(self, target):
        info = process.get_process_info(target.process.pid, ['rss', 'vsz'], ignore_errors=False)
        return {
            'rss': int(info['rss']),
            'vsz': int(info['vsz'])
        }

    def shoot(self, target, plan, session, name):
        self.logger.info('start session %s' % session)
        plugins = [
            tank_executor.JsonReportPlugin(self),
            tank_executor.LunaparkPlugin(self),
            tank_executor.TelegrafPlugin(self),
            tank_executor.OfflinePlugin(self)
        ]
        executor = tank_executor.TankExecutor2(
            tank_executor.DolbiloPlugin(self, plan, "localhost", target.port),
            *plugins
        )
        executor._virtualenv_dir = self.virtualenv_path
        stats_resource, artifacts_dir = self._shoot(executor, name, session)
        self.logger.info('stats_resource: %s' % stats_resource)
        self.logger.info('artifacts_dir: %s' % artifacts_dir)
        stats_results = tank_executor.DolbiloPlugin.get_dumper_stats(artifacts_dir)
        stats_results.update(self._memory_usage(target))
        stats_results.update({
            "report_resource": stats_resource.id,
            "report_path": os.path.relpath(tank_executor.TankExecutor2.get_report_path(artifacts_dir), str(stats_resource.path)),
        })
        lunapark_url = tank_executor.LunaparkPlugin.get_job_url(self)
        if lunapark_url:
            stats_results["report_url"] = lunapark_url
        # Additional fields for the footer.
        stats_results['session'] = session
        stats_results['fail_rate'] = float(stats_results['5xx_requests']) / float(stats_results['total_requests'])
        stats_results['latency_50'] = stats_results['latency_0.5']
        stats_results['latency_95'] = stats_results['latency_0.95']
        self.logger.info('end session %s' % session)
        return stats_results

    def on_execute(self):
        self.logger = logging.getLogger('addrs')
        self.basesearch1 = self.get_addrs_basesearch(chop_group_suffix(self.Parameters, 'addrs_group1', '1'), id=1)
        self.basesearch2 = self.get_addrs_basesearch(chop_group_suffix(self.Parameters, 'addrs_group2', '2'), id=2)
        self.virtualenv_path = self.init_virtualenv()
        sessions = self.Parameters.dolbilka_executor_sessions
        self.logger.info('sessions: %s' % sessions)
        self.logger.info('start warmup')
        self.plan1 = self.Parameters.custom_plan_id1 or self.Parameters.dolbilo_plan_resource_id
        self.plan2 = self.Parameters.custom_plan_id2 or self.Parameters.dolbilo_plan_resource_id
        with self.basesearch1:
            self.logger.info('start shoot warmup')
            self.Context.stats['warmup'] = [self.shoot(self.basesearch1, self.plan1, 1, 'warmup')]
        stats1 = []
        stats2 = []
        for session in xrange(sessions):
            self.logger.info('start 2')
            with self.basesearch2:
                self.logger.info('start shoot 2')
                stats2.append(self.shoot(self.basesearch2, self.plan2, session, '2'))
            self.logger.info('start 1')
            with self.basesearch1:
                self.logger.info('start shoot 1')
                stats1.append(self.shoot(self.basesearch1, self.plan1, session, '1'))
        self.Context.stats['1'] = stats1
        self.Context.stats['2'] = stats2

        # Average top half to avoid down outliers and average rps jumps over time.
        middle = sessions / 2
        avr_count = float(sessions - middle)
        rps1 = sum(sorted([run['rps'] for run in stats1])[middle:]) / avr_count
        rps2 = sum(sorted([run['rps'] for run in stats2])[middle:]) / avr_count

        self.Parameters.description = "{:0.2f}, {:0.2f} | ".format(rps1, rps2) + self.Parameters.description
