# -*- coding: utf-8 -*-

import os
import tarfile
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects.geosearch.tools.nanny import Nanny
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.release_machine.components import all as rmc
import sandbox.projects.release_machine.core.task_env as task_env
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.geosearch import resource_types as geo_types
import sandbox.projects.resource_types as rtypes


class AddrsCollectRMData(sdk2.Task):
    """ Collects data for release-machine jobs """

    class Parameters(sdk2.task.Parameters):
        component_name = sdk2.parameters.String('Component name')
        branch = sdk2.parameters.String('Branch')

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    def last_resource(self, resourse_type, released='stable', additional_attrs={}):
        if released:
            attrs = {'released': released}
        else:
            attrs = additional_attrs
        resource = sdk2.Resource[resourse_type].find(attrs=attrs).first()
        if resource is not None:
            return resource.id
        else:
            return 1359534400

    def from_production(self, service, resource_type):
        token = sdk2.Vault.data('robot-geosearch', 'ADDRS')
        nanny = Nanny(token)
        runtime_attrs = nanny.get_sandbox_files(service)
        sandbox_files = runtime_attrs['sandbox_files']
        for sandbox_file in sandbox_files:
            if sandbox_file.get('resource_type') == resource_type:
                build_task_id = sandbox_file.get('task_id')
                build_task = sdk2.Task[build_task_id]
        resource = sdk2.Resource[resource_type].find(task=build_task).first()
        return resource.id

    def shardmap_from_production(self):
        token = sdk2.Vault.data('robot-geosearch', 'ADDRS')
        nanny = Nanny(token)
        runtime_attrs = nanny.get_sandbox_files('addrs_base')
        shardmap_resource = runtime_attrs['sandbox_bsc_shard']['sandbox_shardmap']
        shardmap_task_id = shardmap_resource['task_id']
        shardmap_task = sdk2.Task[shardmap_task_id]
        shardmap = sdk2.Resource['ADDRS_BUSINESS_SHARDMAP'].find(task=shardmap_task).first()
        return shardmap.id

    def get_startrek_issue(self):
        if self.Parameters.component_name and self.Parameters.branch:
            startrek_token = sdk2.Vault.data('robot-geosearch', 'robot_geosearch_startrek_token')
            startrek_helper = STHelper(startrek_token, useragent='robot-geosearch')
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            release_num = int(self.Parameters.branch)
            issue = startrek_helper.find_ticket_by_release_number(release_num, c_info, fail=False)
            return issue.key

    def _extract_configs(self, archive, files):

        def config_files(members, targets):
            for tarinfo in members:
                if tarinfo.name in targets:
                    yield tarinfo

        tar = tarfile.open(archive)
        tar.extractall(members=config_files(tar, files))
        tar.close()

    def prepare_metasearch_configs(self):
        output_resources = {}
        config_builder_resource = sdk2.Resource[geo_types.GEOMETASEARCH_CONFIGS_BUILDER].find(attrs={'released': 'stable'}).first()
        config_builder = str(sdk2.ResourceData(config_builder_resource).path)
        data = {
            'upper': 'upper.cfg',
            'middle': 'middle.cfg'
        }
        cmd = '{} -i addrs{} -u 8031 -m 8032 -c {}'
        for itype, config_file in data.iteritems():
            log_path = os.path.join(get_logs_folder(), itype + '_build.log')
            with open(log_path, 'w') as log:
                try:
                    subprocess.call(
                        cmd.format(config_builder, itype, config_file),
                        shell=True,
                        stderr=subprocess.STDOUT,
                        stdout=log
                    )
                    current_config_resource = sdk2.Resource[rtypes.GEOMETASEARCH_CONFIG]
                    config_resource = current_config_resource(
                        self,
                        'Geometasearch {} config'.format(itype),
                        config_file
                    )
                    config_resource.config_type = itype
                    config_resource.ttl = 1
                    config_data = sdk2.ResourceData(config_resource)
                    config_data.ready()
                    output_resources.update({itype: config_resource.id})
                except subprocess.CalledProcessError:
                    raise SandboxTaskFailureError('Failed to make config for {}'.format(itype))
        return output_resources

    def on_execute(self):
        self.Context.startek_tiket = self.get_startrek_issue()
        if self.Parameters.component_name == 'begemot':
            self.Context.REMOTE_WIZARD = self.last_resource('REMOTE_WIZARD', additional_attrs={'geosearch_wizard': 'True'})
            self.Context.WIZARD_SHARD = self.last_resource('WIZARD_SHARD', additional_attrs={'geosearch_wizard': 'True'})
            self.Context.WIZARD_GEO_CONFIG_NEW = self.last_resource('WIZARD_GEO_CONFIG_NEW', additional_attrs={'geosearch_wizard': 'True'})
            self.Context.BEGEMOT_GEO_EXECUTABLE = self.last_resource('BEGEMOT_GEO_EXECUTABLE')
            self.Context.BEGEMOT_ISS_GEO = self.last_resource('BEGEMOT_ISS_GEO')
            self.Context.BEGEMOT_CONFIG = self.last_resource('BEGEMOT_CONFIG')
        if self.Parameters.component_name == 'addrs_base':
            self.Context.fast_features_all_shards_resource_id = self.last_resource('MAPS_DATABASE_FAST_FEATURES_SHARD')
            self.Context.geobasesearch_config_resource_id = self.from_production('addrs_base', 'MAPS_SEARCH_CONFIG_BASE')
            self.Context.maps_search_advert_db_resource_id = self.last_resource('MAPS_DATABASE_ADVERT')
            self.Context.binary1 = self.from_production('addrs_base', 'GEOBASESEARCH_EXECUTABLE')
            self.Context.shardmap = self.shardmap_from_production()
        if self.Parameters.component_name == 'addrs_meta':
            self.Context.middlesearch_executable_resource_id = self.from_production('addrs_upper', 'GEOMETASEARCH_EXECUTABLE')
            metasearch_configs = self.prepare_metasearch_configs()
            self.Context.middlesearch_config_resource_id = metasearch_configs.get('middle')
            self.Context.upper_config_resource_id = metasearch_configs.get('upper')
            self.Context.middlesearch_data_resource_id = self.last_resource('MIDDLESEARCH_GEO_DATA', released=False)
        if self.Parameters.component_name == 'geosuggest':
            self.Context.geosuggest_data_resource_id = self.last_resource('GEO_SUGGEST_DATA')
            self.Context.data_builder = self.last_resource('MAPS_GEO_SUGGEST_DATA_BUILDER')
            self.Context.data_from_prod = self.from_production('suggest_maps_yp', 'GEO_SUGGEST_DATA')
