# -*- coding: utf-8 -*-

import os
import json
import time
import jinja2
import logging
import subprocess

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.geosearch.resource_types as geort
import sandbox.common.types.resource as ctr
from sandbox import common
from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder

TIME_BETWEEN_RETRIES = 120
MAX_RUNS = 5


class ITestSuite(object):
    def __init__(self):
        self.stand_url = None
        self.tests_tag = None
        self.task_id = None
        self.rerun = False

    def set_details(self, details):
        pass

    def execute(self):
        raise NotImplementedError()


class BinaryPytestSuite(ITestSuite):
    def __init__(self, executable):
        self._executable = executable

    def run(self, attempt, tests):
        res_name = os.path.join(get_logs_folder(), 'acceptance.res.{}.json'.format(attempt))
        log_name = os.path.join(get_logs_folder(), 'acceptance.out.{}.txt'.format(attempt))

        cmd = [str(self._executable), '--json', res_name, '--metasearch', self.stand_url]
        for test in tests:
            cmd.extend(['-F', test])

        with open(log_name, 'w') as logf:
            p = subprocess.Popen(cmd, stdout=logf, stderr=subprocess.STDOUT)
            p.wait()

        with open(res_name) as resf:
            data = json.load(resf)

        details = {
            'details': data['details'],
            'task_id': self.task_id,
        }
        if attempt == 0:
            details['stats'] = data['stats']
        self.set_details(details)
        return data

    def get_failed_tests(self, res):
        failed_tests = res.get('failedTests', [])
        logging.info('Failed tests: %d', len(failed_tests))
        return failed_tests

    def execute(self):
        run = 0
        res = self.run(run, [])

        if self.rerun:
            while True:
                run += 1
                failed_tests = self.get_failed_tests(res)
                if not failed_tests or run > MAX_RUNS:
                    break
                time.sleep(TIME_BETWEEN_RETRIES)
                logging.info('Trying to re-run tests. Run #%d', run)
                res = self.run(run, failed_tests)


class LastGeosearchAcceptanceExecutableResource(sdk2.parameters.Resource):
    resource_type = geort.GEOSEARCH_ACCEPTANCE_EXECUTABLE

    @common.utils.classproperty
    def default_value(cls):
        resource = sdk2.Resource.find(type=cls.resource_type, state=ctr.State.READY).first()
        return resource.id if resource else None


class AddrsMapkitAcceptanceSDK2(sdk2.Task):
    """Run mapkit acceptance tests on addrs stand"""

    class Parameters(sdk2.task.Parameters):
        stand_url = sdk2.parameters.Url(
            'Stand URL',
            default_value='http://addrs-testing.search.yandex.net/search/stable/yandsearch?',
            required=True
        )
        executable = LastGeosearchAcceptanceExecutableResource(
            'Geosearch acceptance binary executable',
            required=True
        )
        tests_tag = sdk2.parameters.String(
            'Behave tests tag',
            description='Execute tests with specific tag. Running all if empty'
        )
        rerun = sdk2.parameters.Bool(
            'Re-run failed cases',
            default_value=False)

    class Requirements(sdk2.Task.Requirements):
        environment = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass

    @property
    def footer(self):
        if not self.Context.details:
            return 'No data yet'

        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), autoescape=True)
        data_to_render = self.Context.details
        return env.get_template("footer.html").render(data_to_render)

    def on_execute(self):
        suite = BinaryPytestSuite(sdk2.ResourceData(self.Parameters.executable).path)

        def _set_details(details):
            self.Context.details = details
        suite.set_details = _set_details

        suite.stand_url = self.Parameters.stand_url
        suite.tests_tag = self.Parameters.tests_tag
        suite.task_id = self.id
        suite.rerun = self.Parameters.rerun

        suite.execute()
