# -*- coding: utf-8 -*-

import os
import jinja2
import logging

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.geosearch.AddrsMapkitAcceptanceSDK2 import AddrsMapkitAcceptanceSDK2 as mapkit_test


class AddrsMapkitCompareTest(sdk2.Task):
    """Run mapkit acceptance tests on addrs stand"""

    class Parameters(sdk2.task.Parameters):
        reference_stand_url = sdk2.parameters.Url('Reference stand URL',
                                                  required=True)
        test_stand_url = sdk2.parameters.Url('Test stand URL',
                                             required=True)
        tests_tag = sdk2.parameters.String('Behave tests tag',
                                            description='Execute tests with specific tag. Running all if empty')
        startrek_task = sdk2.parameters.String('Startrek task',
                                               required=False)
        with sdk2.parameters.String('Launch type') as launch_type_choices:
            launch_type_choices.values['DB'] = 'Database'
            launch_type_choices.values['RM'] = 'Release machine'
        release_number = sdk2.parameters.Integer('Release number')
        component_name = sdk2.parameters.String('Component name')

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def _make_render_data(self):
        data = {}
        if self.Context.test_details:
            data.update({'test_details': self.Context.test_details})
        if self.Context.diff:
            data.update({'diff': self.Context.diff})
        data.update({'ref': self.Parameters.reference_stand_url,
                     'test': self.Parameters.test_stand_url})
        data['task_id'] = self.Context.test
        return data

    @property
    def footer(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path), autoescape=True)
        data_to_render = self._make_render_data()
        return env.get_template("footer.html").render(data_to_render)

    def write_to_startrek(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path))
        data_to_render = self._make_render_data()
        logging.info('Data for rendering startrek comment: %s' % data_to_render)
        comment = env.get_template("startrek.tpl").render(data_to_render)
        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        if self.Parameters.launch_type_choices == 'RM':
            startrek_helper = STHelper(startrek_token)
            c_info = rmc.COMPONENTS[self.Parameters.component_name]()
            relese_num = self.Parameters.release_number
            startrek_helper.comment(relese_num,
                                    comment,
                                    c_info)
        elif self.Parameters.startrek_task and self.Parameters.launch_type_choices == 'DB':
            self.startrek = StartrekClient(startrek_token)
            self.startrek.add_comment(self.Parameters.startrek_task, comment)

    def count_difference(self):
        reference_task = sdk2.Task[self.Context.reference]
        test_task = sdk2.Task[self.Context.test]
        reference_data = reference_task.Context.details
        test_data = test_task.Context.details
        self.Context.test_details = test_data
        reference_set = set(reference_data['details'].keys())
        test_set = set(test_data['details'].keys())
        diff_keys = test_set.difference(reference_set)
        diff = {key: test_data['details'][key] for key in diff_keys}
        return diff

    def on_execute(self):
        logging.info(self.Parameters.launch_type_choices)
        logging.info(self.Parameters.__dict__)
        if not self.Context.reference and not self.Context.test:
            reference_launch = mapkit_test(self,
                                           description='Mapkit tests on {stand_url} (#{parent_id} subtask)'.format(stand_url=self.Parameters.reference_stand_url, parent_id=self.id),
                                           owner=self.owner,
                                           stand_url=self.Parameters.reference_stand_url,
                                           tests_tag=self.Parameters.tests_tag,
                                           rerun=True,
                                           create_sub_task=True).enqueue()
            self.Context.reference = reference_launch.id
            test_launch = mapkit_test(self,
                                      description='Mapkit tests on {stand_url} (#{parent_id} subtask)'.format(stand_url=self.Parameters.test_stand_url, parent_id=self.id),
                                      owner=self.owner,
                                      stand_url=self.Parameters.test_stand_url,
                                      tests_tag=self.Parameters.tests_tag,
                                      rerun=True,
                                      create_sub_task=True).enqueue()
            self.Context.test = test_launch.id
            raise sdk2.WaitTask([self.Context.reference, self.Context.test],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)
        else:
            for task in self.find():
                if task.status != ctt.Status.SUCCESS:
                    raise SandboxTaskFailureError('Some of mapkit tests failed')
        self.Context.diff = self.count_difference()
        self.write_to_startrek()
