# -*- coding: utf-8 -*
import os
import logging
import subprocess
import json

from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import make_folder
import sandbox.projects.resource_types as rtypes
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosearch import resource_types as geotypes


class AddrsObjectsIndexBuild(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''
        Build objects index
    '''

    class Parameters(sdk2.task.Parameters):
        companies_table = sdk2.parameters.String('Companies table',
                                                 default_value='//home/sprav/altay/prod/snapshot/company',
                                                 required=True)
        rubrics_table = sdk2.parameters.String('Rubrics table',
                                               default_value='//home/sprav/altay/prod/snapshot/rubric',
                                               required=True)
        objects_table = sdk2.parameters.String('Objects table',
                                               default_value='//home/sprav/goods_and_prices/temp/data_4prod_flatten',
                                               required=True)
        yt_pool = sdk2.parameters.String('YT pool for executing operations')

        with sdk2.parameters.Group('Configs and Binaries') as binaries:
            indexer_executable = sdk2.parameters.Resource('Objects indexer executable',
                                                          resource_type=geotypes.GEOSEARCH_OBJECTS_INDEXER)
            panther_idx_cnv_executable = sdk2.parameters.Resource('Panther idx converter executable',
                                                                  resource_type=geotypes.GEOSEARCH_PANTHER_IDX_CONVERTER)
            panther_sq_executable = sdk2.parameters.Resource('Panther converter executable',
                                                             resource_type=geotypes.GEOSEARCH_PANTHER_SQ_CONVERTER)
            panther_attr_sq_executable = sdk2.parameters.Resource('Panther attr converter executable',
                                                                  resource_type=geotypes.GEOSEARCH_PANTHER_ATTR_SQ_CONVERTER)
        with sdk2.parameters.Group('Data sources') as data_sources:
            geo_stat = sdk2.parameters.Resource('Geo.stat',
                                                resource_type=rtypes.GEO_STAT)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        environments = (environments.PipEnvironment('yandex-yt'),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def make_indexer_config(self, result_path):
        cfg = {}
        index_path = os.path.abspath(result_path)
        cfg['OutDir'] = index_path
        cfg['ObjectsTable'] = self.Parameters.objects_table
        cfg['CompaniesTable'] = self.Parameters.companies_table
        cfg['RubricsTable'] = self.Parameters.rubrics_table
        cfg_path = 'indexer.cfg'
        with open(cfg_path, 'w') as config_file:
            config_file.write(json.dumps(cfg))

        make_folder(index_path)
        copy_path(str(sdk2.ResourceData(self.Parameters.geo_stat).path),
                  os.path.join(index_path, 'geo.stat'))
        return cfg_path

    def _exec(self, cmd, log_file_path):
        logging.info('Running: %s' % cmd)
        with open(log_file_path, 'w') as log_file:
            try:
                subprocess.check_call(cmd,
                                      shell=True,
                                      env=self.proc_env,
                                      stdout=log_file,
                                      stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError as err:
                logging.info('%s command failed' % cmd)
                logging.info('Details %s:\n' % err)
                raise SandboxTaskFailureError('Building database failed')

    def on_execute(self):
        self.proc_env = os.environ.copy()
        yt_token = sdk2.Vault.data('GEOMETA-SEARCH', 'yt-token')
        self.proc_env['YT_TOKEN'] = yt_token
        self.proc_env['YT_RETRY_COUNT'] = '50'
        if self.Parameters.yt_pool is not None:
            self.proc_env['YT_POOL'] = self.Parameters.yt_pool

        with self.memoize_stage.BUILD_INDEX(commit_on_entrance=False):
            result_path = './res'
            cfg = self.make_indexer_config(result_path)
            cmd = ('{binary} {cfg}').format(binary=sdk2.ResourceData(self.Parameters.indexer_executable).path,  cfg=cfg)
            log_file_path = get_logs_folder() + '/build_index.out.txt'
            self._exec(cmd, log_file_path)

            cwd = os.getcwd()
            os.chdir(result_path)
            # cmd = ('{binary} -CP --no-default-index-files --shard-path . -f0.1 -vv --index-file ./index:TextIndex').format(binary=sdk2.ResourceData(self.Parameters.panther_idx_cnv_executable).path)
            # log_file_path = get_logs_folder() + '/panther_idx_cnv.out.txt'
            # self._exec(cmd, log_file_path)

            # cmd = ('{binary} -g -i indexpanther -d . -t objects -o indexpanther.offroad').format(binary=sdk2.ResourceData(self.Parameters.panther_sq_executable).path)
            # log_file_path = get_logs_folder() + '/panther_sq_cnv.out.txt'
            # self._exec(cmd, log_file_path)

            # cmd = ('{binary} -i index -d . -t objects').format(binary=sdk2.ResourceData(self.Parameters.panther_attr_sq_executable).path)
            # log_file_path = get_logs_folder() + '/panther_attr_sq_cnv.out.txt'
            # self._exec(cmd, log_file_path)

            # for suffix in ['fat', 'fat.idx', 'info', 'inv', 'inv.model', 'key', 'key.model']:
            #    os.remove('indexpanther.' + suffix)

            os.chdir(cwd)
            resource = sdk2.Resource[geotypes.GEOSEARCH_OBJECTS_INDEX]
            current_resource = resource(self, 'objects basesearch index', result_path)
            data = sdk2.ResourceData(current_resource)
            data.ready()
