# -*- coding: utf-8 -*-

import os
import math
import json
import jinja2
import logging
import requests
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk import environments
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.geosearch.AddrsMapkitAcceptanceSDK2 import AddrsMapkitAcceptanceSDK2


class AddrsRatingsTest(AddrsMapkitAcceptanceSDK2):
    """Run rating test tests on addrs stand"""

    class Parameters(sdk2.task.Parameters):
        reference_stand_url = sdk2.parameters.Url('Reference stand URL',
                                                  required=True)
        test_stand_url = sdk2.parameters.Url('Test stand URL',
                                             required=True)
        behave_arcadia_url = sdk2.parameters.ArcadiaUrl('Behave framework URL',
                                                        default_value='arcadia:/arc/trunk/arcadia/search/geo/tools/geosearch_acceptance',
                                                        required=True)
        startrek_task = sdk2.parameters.String('Startrek task',
                                               required=False)

    class Requirements(sdk2.Task.Requirements):
        cores = 1  # exactly 1 core
        ram = 8192  # 8GiB or less

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_script(self):
        checkout_path = Arcadia.get_arcadia_src_dir('arcadia:/arc/trunk/arcadia/search/geo/tools/production')
        return os.path.join(checkout_path, 'get_rating.py')

    def get_oids(self):
        current_dir = os.path.dirname(os.path.abspath(__file__))
        permalinks_path = os.path.join(current_dir, 'permalinks')
        with open(permalinks_path) as f:
            oids = f.readlines()
        oids = '&business_oid=%s' % '&business_oid='.join(oids)
        logging.info('oids: %s' % oids)
        return oids

    def get_deviation(self, url):
        from pb_response_wrapper import PbSearchResult as SearchResult
        cgi = '&lang=ru_RU&origin=1&add_snippet=businessrating/2.x&gta=businessrating/2.x'
        oids = self.get_oids()
        full_url = url + cgi + oids
        response = requests.get(full_url)
        results = SearchResult(response.content)
        ratings = [go._rating_metadata().score for go in results.geo_objects]
        logging.info('Ratings for %s:\n%s' % (full_url, ratings))
        deviation = math.sqrt(sum([pow(rating, 2) for rating in ratings]))
        return deviation

    def write_to_startrek(self):
        template_path = os.path.dirname(os.path.abspath(__file__))
        env = jinja2.Environment(loader=jinja2.FileSystemLoader(template_path),
                                 extensions=['jinja2.ext.do'])
        data_to_render = {'stable': self.Context.stable_deviation,
                          'test': self.Context.test_deviation,
                          'task': self.id}
        logging.info('Data for rendering startrek comment: %s' % data_to_render)
        comment = env.get_template("startrek.tpl").render(data_to_render)
        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        self.startrek = StartrekClient(startrek_token)
        self.startrek.add_comment(self.Parameters.startrek_task, comment)

    def on_execute(self):
        self.Context.tests_path = self.get_tests()
        self.protoc = self.get_protoc()
        self.tests_path = self.Context.tests_path
        self.proto_path = '%s/proto/search/' % self.tests_path
        self.compile_protobuf()
        try:
            with environments.VirtualEnvironment() as venv:
                venv.pip('pip==9.0.1')
                venv.pip('behave==1.2.5')
                venv.pip('-i https://pypi.yandex-team.ru/simple requests')
                venv.pip('-i https://pypi.yandex-team.ru/simple protobuf==3.3.0')
                cmd_tpl = ('{python} {script} -u "{stand}" -o "{oids}" -d "{dir}"')
                os.chdir(os.path.join(self.tests_path, 'steps/stand'))
                stable = subprocess.check_output(cmd_tpl.format(python=venv.executable,
                                                                script=self.Context.script,
                                                                stand=self.Parameters.reference_stand_url,
                                                                oids=self.get_oids(),
                                                                dir=os.path.join(self.tests_path, 'steps/stand')),
                                                 shell=True)
                test = subprocess.check_output(cmd_tpl.format(python=venv.executable,
                                                              script=self.Context.script,
                                                              stand=self.Parameters.test_stand_url,
                                                              oids=self.get_oids(),
                                                              dir=os.path.join(self.tests_path, 'steps/stand')),
                                                 shell=True)
            logging.info('stable: %s' % stable)
            logging.info('test: %s' % test)
            self.Context.stable_deviation = json.loads(stable)
            self.Context.test_deviation = json.loads(test)
            with self.memoize_stage.STARTREK(commit_on_entrance=False):
                self.write_to_startrek()
        except subprocess.CalledProcessError as err:
            logging.info('Calculating ratings failed')
            logging.info('Details: %s' % err)
