import logging
from collections import OrderedDict

import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects.common.geosearch.utils_sdk2 import make_parameters_cls
from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types as rt
from sandbox.projects.geosearch import resource_types as grt
from sandbox.projects.geosearch.AddrsBaseDatabaseYtBuild import AddrsBaseDatabaseYtBuild
from sandbox.projects.geosearch.tools.database_notifications import notify_by_telegram
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class AddrsBaseFreshDatabaseYtBuild(AddrsBaseDatabaseYtBuild):

    class Parameters(sdk2.Parameters):
        fresh_timestamp = sdk2.parameters.Integer('Geosearch index creation time', default=0)
        extra = AddrsBaseDatabaseYtBuild.Parameters

    def cut_export(self, export_path, output_path):
        if not self.Parameters.fresh_timestamp or self.Parameters.fresh_timestamp <= 0:
            return False
        cmd = ('cut_export '
               '-d {output_path} '
               '-s hahn '
               '-i {export_path} '
               '-t {timestamp}').format(output_path=output_path,
                                        export_path=export_path,
                                        timestamp=self.Parameters.fresh_timestamp)
        log_file_path = get_logs_folder() + '/cut_export.out.txt'
        self.run_sync(cmd, log_file_path, 'Cutting companies table failed')
        return True

    def on_release(self, params):
        logging.debug("Release parameters: %r", params)
        release_dict = dict(releaser=params['releaser'],
                            release_status=params['release_status'],
                            release_subject=params['release_subject'],
                            email_notifications=dict(to=[], cc=[]),
                            release_comments=params['release_comments'])
        nanny.ReleaseToNannyTask2.on_release(self, release_dict)
        self.mark_released_resources(params["release_status"], ttl=7)


HIDDEN_BUILD_PARAMS = [
    'need_acceptance',
    'shards_count',
    'release_type',
    'mark_index',
    'tier_name',
    'yt_table',
    'companies_limit',
    'export_svd',
    'export_canonizer',
    'create_sub_task',
    'fresh_timestamp',
]


def _is_resource(parameter):
    return parameter.__name__ == 'Resource'


def _extract_parameters(task_cls):
    for name in task_cls.Parameters.__custom_parameters_names__:
        parameter = getattr(task_cls.Parameters, name)
        if _is_resource(parameter):
            yield name, parameter()
        else:
            yield name, parameter(default=None, required=False)


def get_parameters_dict():
    parameters_dict = OrderedDict()

    for name, parameter in _extract_parameters(AddrsBaseFreshDatabaseYtBuild):
        if name not in HIDDEN_BUILD_PARAMS:
            parameters_dict[name] = parameter
    return parameters_dict


class BuildAddrsFreshShard(sdk2.Task):
    '''
    Build geobasesearch search freshness shard
    '''
    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    with sdk2.parameters.String('Release the build to') as release_type:
        release_type.values[''] = release_type.Value(value='do not release', default=True)
        for val in ctt.ReleaseStatus:
            release_type.values[val] = release_type.Value(value=val)

    Parameters = make_parameters_cls(OrderedDict(
        [
            ('common_params', sdk2.parameters.Group('Common parameters')),
            ('release_type', release_type),
            ('release_subject', sdk2.parameters.String('Release subject', default='Search freshness shard')),
            ('release_comments', sdk2.parameters.String('Release comments')),
            ('fresh_export_path', sdk2.parameters.String('Fresh states YT table', default='//home/sprav/fast_export/fresh_states', required=True)),
            ('check_timestamp', sdk2.parameters.Bool('Check geobasesearch index creation time', default=False)),
            ('sharded', sdk2.parameters.Bool('Build sharded freshness (2 shards)', default=False)),

            ('build_task_params', sdk2.parameters.Group('Optional build task parameters')),
        ] +
        list(get_parameters_dict().iteritems())
    ))

    def on_failure(self, prev_status):
        notify_by_telegram(self, 'failed')

    def on_break(self, prev_status, status):
        notify_by_telegram(self, 'failed with exception')

    def on_timeout(self, prev_status):
        notify_by_telegram(self, 'timed out')

    def on_execute(self):
        with self.memoize_stage.run_build(commit_on_entrance=False):
            stable_shardmap = sdk2.Resource[rt.ADDRS_BUSINESS_SHARDMAP].find(attrs={'released': 'stable'}).first()
            logging.info('selected shardmap: {0}'.format(stable_shardmap.id))
            stable_index_build_task = sdk2.Task[stable_shardmap.task_id]
            logging.info('selected build task: {0}'.format(stable_shardmap.task_id))

            parameters = dict(stable_index_build_task.Parameters)
            parameters['yt_table'] = self.Parameters.fresh_export_path
            parameters['shards_count'] = (2 if self.Parameters.sharded else 1)
            parameters['need_acceptance'] = False
            parameters['export_svd'] = False
            parameters['export_canonizer'] = False
            parameters['create_sub_task'] = True
            parameters['companies_limit'] = 0
            parameters['ttl_days'] = 1
            parameters['kill_timeout'] = self.Parameters.kill_timeout
            if self.Parameters.check_timestamp:
                parameters['fresh_timestamp'] = stable_index_build_task.Context.cooking_start_time - 1800

            precomputed_filters_resource = sdk2.Resource[rt.BUSINESS_COMPUTED_FILTERS].find(task=stable_index_build_task).first()
            if precomputed_filters_resource:
                parameters['precomuted_filters'] = precomputed_filters_resource

            moved_collapser_dump_resource = sdk2.Resource[grt.GEOSEARCH_MOVED_COLLAPSER_DUMP].find(task=stable_index_build_task).first()
            if moved_collapser_dump_resource:
                parameters['moved_collapser_dump'] = moved_collapser_dump_resource

            for name in parameters:
                if name in HIDDEN_BUILD_PARAMS:
                    continue
                value = getattr(self.Parameters, name)
                if value:
                    parameters[name] = value

            fresh_build_task_class = sdk2.Task[AddrsBaseFreshDatabaseYtBuild.type]
            fresh_build_task = fresh_build_task_class(stable_index_build_task.current, **parameters)
            if stable_index_build_task.Context.search_text_table is not None:
                fresh_build_task.Context.search_text_table = stable_index_build_task.Context.search_text_table
            fresh_build_task.save()
            fresh_build_task.enqueue()
            self.Context._FRESH_BUILD_TASK_KEY = fresh_build_task.id

            raise sdk2.WaitTask([self.Context._FRESH_BUILD_TASK_KEY],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)

        build_task = sdk2.Task[self.Context._FRESH_BUILD_TASK_KEY]
        bad_statuses = ['FAILURE',
                        'EXCEPTION',
                        'TIMEOUT',
                        'STOPPED']
        if build_task.status in bad_statuses:
            msg = 'Build index failed'
            raise SandboxTaskFailureError(msg)

        self.Context.register_shard_task_id = build_task.Context.register_tasks[0]

        if self.Parameters.release_type:
            self.server.release(task_id=(build_task.id if self.Parameters.sharded else self.Context.register_shard_task_id),
                                type=self.Parameters.release_type,
                                subject=self.Parameters.release_subject,
                                comments=self.Parameters.release_comments)
