# -*- coding: utf-8 -*

import time
import tarfile
import logging
import requests
import datetime
from sandbox import sdk2

import sandbox.common.types.task as ctt
from sandbox.common.types.misc import NotExists
from sandbox.projects.geosearch.tools import nirvana
from sandbox.projects import resource_types as rtypes
from sandbox.projects.geosearch import resource_types as geotypes
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosearch import resource_types as geo_types
from sandbox.projects.geosearch.ParseAddrsUrlsFromYt import ParseAddrsUrlsFromYt


def get_date():
    return datetime.datetime.now().strftime('%Y-%m-%d %H:%M')


class BuildAddrsWebIndexann(sdk2.Task):
    '''
        Build MAPS_WEB_INDEXANN
    '''

    class Parameters(sdk2.task.Parameters):
        reference_workflow = sdk2.parameters.String(
            'Reference workflow id',
            required=True,
            # default_value='69414464-5749-411f-9966-86c5e75e642b')
            default_value='8daa0170-1a66-4c4c-8f9a-51981525a5b5'
        )
        target_block_name = sdk2.parameters.String(
            'Target block name',
            default_value='[SURL] Build annotation index'
        )
        doublefrc_block_name = sdk2.parameters.String(
            'doublefrc.zz block name',
            default_value='Filter web annotations'
        )
        company_table_path = sdk2.parameters.String(
            'Company table path',
            required=True
        )
        url_parser = sdk2.parameters.Resource(
            'Geosearch YT url parser',
            resource_type=geo_types.GEOSEARCH_YT_URL_PARSER,
            required=True
        )
        url_filter = sdk2.parameters.Resource(
            'WebAnn url filter',
            resource_type=geo_types.GEOSEARCH_WEB_INDEXANN_URL_FILTER
        )

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    @sdk2.header()
    def head(self):
        if not type(self.Context.new_workflow) == NotExists:
            url = 'https://nirvana.yandex-team.ru/flow/%s/graph' % self.Context.new_workflow
            return '<a href="{url}">{url}</a>'.format(url=url)
        else:
            return ''

    def make_urls(self):
        if not self.Context.build_business_urls_task:
            parse_urls_task_class = sdk2.Task[ParseAddrsUrlsFromYt.type]
            parse_urls_task = parse_urls_task_class(self,
                                                    description='Build MAPS_WEB_URLS_BUSINESS from YT',
                                                    company_table=self.Parameters.company_table_path,
                                                    url_parser_executable=self.Parameters.url_parser,
                                                    owner=self.owner,
                                                    create_sub_task=True)
            parse_urls_task.enqueue()
            self.Context.build_business_urls_task = parse_urls_task.id
            raise sdk2.WaitTask([self.Context.build_business_urls_task],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)
        else:
            task = sdk2.Task[self.Context.build_business_urls_task]
            if task.status in ctt.Status.Group.BREAK:
                raise SandboxTaskFailureError('Parsing URLs failed')

    def on_execute(self):
        with self.memoize_stage.parse_urls:
            self.make_urls()
        parse_urls_task = sdk2.Task[self.Context.build_business_urls_task]
        business_urls_resource = sdk2.Resource[rtypes.MAPS_WEB_URLS_BUSINESS].find(task=parse_urls_task).first()
        logging.info('business_urls_resource => %s' % business_urls_resource)
        wiki_urls_resource = sdk2.Resource[rtypes.MAPS_WEB_URLS_POI].find(attrs={'released': 'stable'}).first()
        nirvana_token = sdk2.Vault.data('robot-geosearch', 'nirvana_token')
        nirvana_cli = nirvana.Nirvana(nirvana_token)
        logging.info('Current workflow: %s' % self.Context.new_workflow)
        if not self.Context.new_workflow:    # Workflow not created yet
            workflow = nirvana_cli.clone_workflow(self.Parameters.reference_workflow,
                                                  'SURL + WebAnn %s' % get_date(),
                                                  'SURL + WebAnn %s' % get_date())
            workflow_params = [
                {'parameter': 'timestamp', 'value': int(time.time())},
                {'parameter': 'business_urls_resource_id', 'value': str(business_urls_resource.id)},
                {'parameter': 'poi_urls_resource_id', 'value': str(wiki_urls_resource.id)}
            ]
            if self.Parameters.url_filter:
                workflow_params.append({'parameter': 'mr_filter_resource_id', 'value': str(self.Parameters.url_filter.id)})
            nirvana_cli.set_workflow_parameters(workflow, workflow_params)
            nirvana_cli.start_workflow(workflow)
            self.Context.new_workflow = workflow
            raise sdk2.WaitTime(600)
        else:
            workflow = self.Context.new_workflow
            while nirvana_cli.get_workflow_state(workflow)['status'] != 'completed':
                raise sdk2.WaitTime(600)
            if nirvana_cli.get_workflow_state(workflow)['result'] == 'failure':
                nirvana_msg = nirvana_cli.get_workflow_state(workflow)['message']
                msg = ('Workflow <a href="https://nirvana.yandex-team.ru/flow/'
                       '{workflow_id}/graph">{workflow_id}</a> failed: '
                       '{workflow_err}')
                raise SandboxTaskFailureError(msg.format(workflow_id=workflow,
                                                         workflow_err=nirvana_msg))
            else:
                if self.Parameters.target_block_name:
                    target_block_id = nirvana_cli.get_block_by_name(self.Context.new_workflow,
                                                                    self.Parameters.target_block_name)
                    storage_path = nirvana_cli.get_target_block_result(self.Context.new_workflow,
                                                                       target_block_id)
                    tar_file_path = './graph_output.tar.gz'
                    download_request = requests.get(storage_path,
                                                    verify=False,
                                                    headers={'Authorization': 'OAuth {}'.format(nirvana_token)})
                    with open(tar_file_path, 'w') as tar_file:
                        tar_file.write(download_request.content)
                    tar = tarfile.open(tar_file_path)
                    result = tar.getnames()[0]
                    tar.extractall()
                    tar.close()
                    web_indexann_resource = sdk2.Resource[rtypes.MAPS_WEB_INDEXANN]
                    current_web_indexann_resource = web_indexann_resource(self,
                                                                          'Web index annotations',
                                                                           result)
                    web_indexann_data = sdk2.ResourceData(current_web_indexann_resource)
                    web_indexann_data.ready()

                if self.Parameters.doublefrc_block_name:
                    doublefrc_block_id = nirvana_cli.get_block_by_name(self.Context.new_workflow,
                                                                       self.Parameters.doublefrc_block_name)
                    doublefrc_path = nirvana_cli.get_target_block_result(self.Context.new_workflow,
                                                                         doublefrc_block_id)
                    download_doublefrc = requests.get(doublefrc_path,
                                                      verify=False,
                                                      headers={'Authorization': 'OAuth {}'.format(nirvana_token)})
                    doublefrc_path = './doublefrc.zz'
                    with open(doublefrc_path, 'w') as result:
                        result.write(download_doublefrc.content)
                    doublefrc_resource = sdk2.Resource[geotypes.MAPS_WEB_DOUBLEFRC]
                    current_doublefrc_resource = doublefrc_resource(self, 'Web index annotations raw data', doublefrc_path)
                    doublefrc_data = sdk2.ResourceData(current_doublefrc_resource)
                    doublefrc_data.ready()
