# -*- coding: utf-8 -*-
import logging
import os
import re
import shutil
import tempfile
import zipfile

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc

from sandbox.projects.common import file_utils as fu


class MapsDebugExtension(sdk2.Resource):
    auto_backup = True
    any_arch = True

    ext_version = sdk2.parameters.String()


class CrxCreator(sdk2.Resource):
    auto_backup = True
    releasable = True


class BuildMapsDebugExtension(sdk2.Task):
    """
        Building yandex.bro extension to debug geosearch at yandex maps
    """

    REQUIRED_FILES = [
        'maps_debug_extension.crx',
        'zhuk2.xml'
    ]

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY | ctc.Tag.LINUX_PRECISE | ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl('Arcadia url', required=True)
        with sdk2.parameters.Group('Debug params') as debug_params:
            crx_creator = sdk2.parameters.Resource('Crx creator', resource_type=[CrxCreator], required=False)
            update_version = sdk2.parameters.Bool('Update extension version', default=True)

    def _checkout(self):
        svn_url = sdk2.svn.Arcadia.svn_url(self.Parameters.arcadia_url)
        svn_url = sdk2.svn.Arcadia.replace(svn_url, 'arc/trunk/arcadia/search/geo/tools/maps_debug_extension')
        logging.info('Svn URL: %s' % svn_url)
        self.checkout_path = sdk2.svn.Arcadia.checkout(svn_url, tempfile.mkdtemp())
        logging.info('Checkout path: %s' % self.checkout_path)

        self.ext_version = self._increment_version_and_patch_manifest(self.Parameters.update_version)
        logging.info('Version: %s' % self.ext_version)
        if self.Parameters.update_version:
            self._patch_update_xml(self.ext_version)

            sdk2.svn.Arcadia.commit(self.checkout_path, "MapsDebugExtension autobuild version " + self.ext_version + " SKIP_CHECK", user='zomb-sandbox-rw')

    def on_execute(self):
        self._checkout()
        self._create_crx()
        self._publish_resource(self.ext_version)

    def _publish_resource(self, ext_version):
        os.mkdir('extension')
        for f in self.REQUIRED_FILES:
            shutil.copy(os.path.join(self.checkout_path, f), 'extension')

        resource = sdk2.ResourceData(MapsDebugExtension(self, "Maps debug extension v{}".format(ext_version), 'extension', ext_version=ext_version))
        resource.ready()

    def _create_crx(self):
        archive = zipfile.ZipFile('extension.zip', 'w', zipfile.ZIP_DEFLATED)
        for root, dirs, files in os.walk(self.checkout_path):
            if '.svn' in root.split('/'):
                continue
            for file in files:
                file_path = os.path.join(root, file)
                archive.write(file_path, os.path.relpath(file_path, self.checkout_path))
        archive.close()

        crx_creator = self.Parameters.crx_creator
        if not crx_creator:
            crx_creator = sdk2.Resource.find(type=CrxCreator, released="stable").order(-sdk2.Resource.id).first()
            if not crx_creator:
                logging.error('Can\'t find crx creator')
                return
        crx_creator_exe = sdk2.ResourceData(crx_creator)

        fu.write_file('key.pem', sdk2.Vault.data('maps_debug_extension_publisher'))

        crx_file = os.path.join(self.checkout_path, 'maps_debug_extension.crx')

        with sdk2.helpers.ProcessLog(self, logger="crx_creator") as pl:
            sp.check_call([str(crx_creator_exe.path), '-i', 'extension.zip', '-k', 'key.pem', '-o', crx_file], stdout=pl.stdout, stderr=pl.stderr)
        os.remove('key.pem')

    def _increment_version_and_patch_manifest(self, update_version):
        # Deduce version
        manifest_path = os.path.join(self.checkout_path, 'manifest.json')
        manifest = fu.json_load(manifest_path)
        ver_split = manifest['version'].split('.')
        if update_version:
            ver_split[-1] = '{}'.format(int(ver_split[-1]) + 1)
        ext_version = '.'.join(ver_split)

        if update_version:
            # Set version at local copy
            manifest['version'] = ext_version
            fu.json_dump(manifest_path, manifest, indent=2)

        return ext_version

    @staticmethod
    def _patch_line(l, ext_version):
        if 'updatecheck' in l:
            return re.sub(r"version='.+'", "version='{}'".format(ext_version), l)

        return l

    def _patch_update_xml(self, ext_version):
        update_path = os.path.join(self.checkout_path, 'zhuk2.xml')
        content = [BuildMapsDebugExtension._patch_line(l, ext_version) for l in fu.read_line_by_line(update_path)]
        fu.write_lines(update_path, content)
