# -*- coding: utf-8 -*-

import os

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter, ResourceSelector, SandboxIntegerParameter, SandboxRadioParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.utils import get_or_default
from sandbox.sandboxsdk.task import SandboxTask

from sandbox.projects import resource_types as rt
from sandbox.projects.geosearch import resource_types as geort
from sandbox.projects.common.geosearch.indexing import BuildHelper
from sandbox.projects.common.geosearch.utils import unpack_file
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask


class FastFeaturesSourceParameter(ResourceSelector):
    name = 'data_source_resource_id'
    description = 'Data source'
    resource_type = rt.MAPS_DATABASE_FAST_FEATURES_SOURCE
    default_url_value = 'http://export.backa.yandex.ru/fast_features.pb.gz'


class FastFeaturesIndexerParameter(LastReleasedResource):
    name = 'indexer_resource_id'
    description = 'Indexer binary'
    resource_type = rt.GEO_FAST_FEATURES_INDEXER_EXECUTABLE


class OutputTable(SandboxStringParameter):
    name = 'output_table'
    description = 'Output table: '
    default_value = ''


class ShardCount(SandboxIntegerParameter):
    name = 'shard_count'
    description = 'Geobasesearch shards count'
    default_value = 1


class OutputResourceType(SandboxRadioParameter):
    name = 'output_resource_type'
    description = 'Output resource type'
    default_value = 'MAPS_DATABASE_FAST_FEATURES'
    choices = [
        ('Obsolete', 'MAPS_DATABASE_FAST_FEATURES'),
        ('Relevant', 'MAPS_DATABASE_FAST_FEATURES_SHARDED'),
    ]
    per_line = 1


class BuildMapsFastFeatures(ReleaseToNannyTask, SandboxTask):
    """
        Build maps fast features index
    """
    type = 'BUILD_MAPS_FAST_FEATURES'

    input_parameters = (FastFeaturesSourceParameter, FastFeaturesIndexerParameter, OutputTable,
                        ShardCount, OutputResourceType)

    def on_execute(self):
        # fetch resources
        resource_id = self.ctx[FastFeaturesSourceParameter.name]
        fast_features_pb_gz = self.sync_resource(resource_id)
        if not os.path.isfile(fast_features_pb_gz):
            raise SandboxTaskFailureError('data source must be single binary proto file (gzipped)')

        source_dir = 'fast_features_source'
        res = make_folder(source_dir)
        unpack_file(fast_features_pb_gz, res, to_name='fast_features.pb')
        fast_features_pb = os.path.join(source_dir, 'fast_features.pb')
        if not os.path.isfile(fast_features_pb):
            raise SandboxTaskFailureError('data source must be single binary proto file')

        indexer = self.sync_resource(self.ctx[FastFeaturesIndexerParameter.name])

        # create empty index directory
        index_directory = 'index'
        make_folder(index_directory)

        # run indexer, it creates index/000X/fast_features.mms
        exec_params = [indexer, '-i', fast_features_pb, '-d', index_directory]
        shard_count = get_or_default(self.ctx, ShardCount)
        if shard_count > 1:
            exec_params.extend(['--shards-count', str(shard_count)])
        out_table = get_or_default(self.ctx, OutputTable)
        if out_table:
            yt_token = self.get_vault_data('mesherin', 'YQL_TOKEN')
            os.environ['YT_TOKEN'] = yt_token
            exec_params.extend(['--output-table', out_table])
        run_process(exec_params, log_prefix='indexer')

        # publish without tar
        resource_type = geort.MAPS_DATABASE_FAST_FEATURES if self.ctx[OutputResourceType.name] == 'MAPS_DATABASE_FAST_FEATURES' else geort.MAPS_DATABASE_FAST_FEATURES_SHARDED
        BuildHelper.create_index_resource(self,
                                          primary_source_resource_id=resource_id,
                                          index_directory=index_directory,
                                          index_type=resource_type,
                                          additional_attributes={"shards_count": str(shard_count)})

        if shard_count > 1:
            # It is not allowed for different resources to have common files,
            # so we make copies of mms files for each shard to share them individually.
            files = self.copy_shard_files(shard_count, index_directory)
            self.share_shards(shard_count, files)

    def copy_shard_files(self, shard_count, index_directory):
        '''
        Copy all mms files from indexer output to root task directory (with no hierarchy).
        Retun a list of paths to new files.
        '''
        result = []
        for i in range(shard_count):
            fn = 'fast_features.mms.{}'.format(i)
            srcpath = os.path.join(index_directory, '0001', fn)
            if not os.path.isfile(srcpath):
                raise SandboxTaskFailureError('shard #{} not found in indexer output (path: "{}")'.format(i, srcpath))
            os.link(srcpath, fn)  # hard link
            result.append(fn)
        return result

    def share_shards(self, shard_count, files):
        '''
        Create a single resource for each shard.
        '''
        for i in range(shard_count):
            attrs = {
                'shard_id': str(i),
                'shards_count': str(shard_count),
            }
            shard = self.create_resource(description=self.descr, resource_path=files[i],
                                         resource_type=geort.MAPS_DATABASE_FAST_FEATURES_SHARD, attributes=attrs)
            self.mark_resource_ready(shard.id)

    def mark_released_resources(self, status, ttl=30):
        return SandboxTask.mark_released_resources(self, status, ttl)

__Task__ = BuildMapsFastFeatures
