import logging
import os
import sys
import traceback

from sandbox import common
from sandbox import sdk2
from sandbox.projects import resource_types as rt
from sandbox.projects.geosearch import resource_types as geort
from sandbox.projects.geosearch.tools import fast_feature
from sandbox.sandboxsdk import environments
from sandbox.sdk2 import yav


class YtProxy(object):
    def __init__(self, proxy, yt_token, fast_feature_output):
        import yt.wrapper as yt
        self.proxy = proxy
        self.fast_feature_output_value = fast_feature_output
        self.client = yt.YtClient(proxy=proxy, token=yt_token)

    @property
    def fast_feature_output(self):
        return self.fast_feature_output_value

    def source_path(self, path):
        assert self.client.exists(path), 'Source {} not found on {}'.format(path, self.proxy)
        return path


class YtFallbackProxy(YtProxy):
    def __init__(self, proxy, yt_token, fallback_dir):
        super(YtFallbackProxy, self).__init__(proxy=proxy, yt_token=yt_token, fast_feature_output=None)
        self.fallback_dir = fallback_dir

    @property
    def fast_feature_output(self):
        if not self.fast_feature_output_value:
            self.fast_feature_output_value = self.client.create_temp_table()
        return self.fast_feature_output_value

    def source_path(self, path):
        fallback_path, fallback_time = fast_feature.fallback_source(self.client, self.fallback_dir, path)
        assert fallback_path and fallback_time, 'Source {} not found on {}'.format(path, self.proxy)
        return fallback_path


class BuildMapsFastFeatures2(sdk2.Task):
    '''
    Build maps fast features index SDK2
    '''

    class Requirements(sdk2.Requirements):
        cores = 1
        ram = 16384

        environments = [
            environments.PipEnvironment('yandex-yt', version='0.10.8'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        yt_token = sdk2.parameters.YavSecret('YT token', default=yav.Secret('sec-01daxf0w5f7c1av3p04jazan1e', default_key='secret'))

        yt_proxy = sdk2.parameters.String('YT proxy', default='hahn', required=True)
        use_fallback = sdk2.parameters.Bool('Use fallback', default=False)
        with use_fallback.value[True]:
            fallback_yt_proxy = sdk2.parameters.String('Fallback YT proxy', default='arnold', required=True)
            fallback_data_path = sdk2.parameters.String('Fallback data path', default='//home/geo-search/fast-feature-fallback', required=True)
        indexer = sdk2.parameters.Resource('Fast features indexer binary', resource_type=rt.GEO_FAST_FEATURES_INDEXER_EXECUTABLE, required=True)
        fast_feature_values = sdk2.parameters.Resource('Fast features updater binary', resource_type=geort.GEOSEARCH_UPDATE_FAST_FEATURE_VALUES, required=False)
        shard_count = sdk2.parameters.Integer('Number of shards', default=1, required=True)
        create_single_resource = sdk2.parameters.Bool('Create single resource which contains all shards', default=False, required=False)
        read_hotel_prices = sdk2.parameters.Bool('Read data for hotel price filter', default=False, required=False)
        build_fresh_references = sdk2.parameters.Bool('Read references from freshness table', default=False, required=False)
        build_fresh_shard = sdk2.parameters.Bool('Build data for search freshness shard', default=False, required=False)
        build_quarantine_fixlists = sdk2.parameters.Bool('Build quarantine fixlists', default=False, required=False)
        fast_features_table = sdk2.parameters.String('Table with fast feature values', default='//home/sprav/fast_features/stable/fast_features', required=False)
        recalc_fast_features = sdk2.parameters.Bool('Recalc fast feature values', default=False, required=False)
        build_advert_chains_data = sdk2.parameters.Bool('Build data for advert chain filter', default=False, required=False)
        companies_table = sdk2.parameters.String('Companies export table', default='//home/altay/db/export/current-state/exported/company', required=False)
        export_path = sdk2.parameters.String('Altay export path', default='//home/altay/db/export/current-state/exported', required=False)
        advert_chains_table = sdk2.parameters.String('Advert chains table', default='//home/geosearch-prod/snippets/input_tables/advert_chain_filter', required=False)
        build_company_prices_data = sdk2.parameters.Bool('Build data for objects_price filter', default=False, required=False)
        maps_goods_export_table = sdk2.parameters.String('Maps goods data_4production table', default='//home/maps/geoapp/goods/stable/db/data_4production', required=False)

    def build(self, yt_proxy, index_directory):
        indexer_path = sdk2.ResourceData(self.Parameters.indexer).path

        if self.Parameters.recalc_fast_features:
            updater_path = sdk2.ResourceData(self.Parameters.fast_feature_values).path
            cmdline = [
                str(updater_path),
                '--yt-proxy', yt_proxy.proxy,
                '--afisha', yt_proxy.source_path('//home/afisha/export/production/schedules'),
                '--pharmacyOnDuty',
                yt_proxy.source_path('//home/sprav/fast_features/stable/pharmacy_on_duty_turkey_partner'),
                '--providers', yt_proxy.source_path('//home/sprav/altay/prod/export/source/fast_feature_providers'),
                '--outputTable', yt_proxy.fast_feature_output,
            ]
            with sdk2.helpers.ProcessLog(self, logger='updater') as pl:
                sdk2.helpers.subprocess.check_call(cmdline, stdout=pl.stdout, stderr=pl.stderr)

        cmdline = [
            str(indexer_path),
            '-y',
            '--proxy', yt_proxy.proxy,
            '--input', yt_proxy.fast_feature_output,
            '--fresh-input', yt_proxy.source_path('//home/sprav/fast_export/fresh_states'),
            '--shards-count', str(self.Parameters.shard_count),
            '--output-dir', index_directory,
        ]
        if self.Parameters.read_hotel_prices:
            cmdline.extend(['--price-input', yt_proxy.source_path(
                '//home/travel/prod/prices/price_filter_data/latest/price_filter_data')])
        if self.Parameters.build_fresh_references:
            cmdline.append('-r')
        if self.Parameters.build_fresh_shard:
            cmdline.append('-f')
        if self.Parameters.build_quarantine_fixlists:
            cmdline.extend(['--rubric-fixlist', yt_proxy.source_path('//home/sprav/quarantine/fixlist_rubrics')])
            cmdline.extend(['--chain-fixlist', yt_proxy.source_path('//home/sprav/quarantine/fixlist_chains')])
        if self.Parameters.build_advert_chains_data:
            cmdline.extend(['--advert', self.Parameters.advert_chains_table,
                            '--export', self.Parameters.export_path,
                            '--companies', self.Parameters.companies_table])
        if self.Parameters.build_company_prices_data:
            cmdline.extend(['--maps-goods-export', self.Parameters.maps_goods_export_table])
        with sdk2.helpers.ProcessLog(self, logger='indexer') as pl:
            sdk2.helpers.subprocess.check_call(cmdline, stdout=pl.stdout, stderr=pl.stderr)

    def on_execute(self):
        yt_token = self.Parameters.yt_token.data()['secret']
        self._set_environment(yt_token)

        index_directory = 'index'
        os.mkdir(index_directory)

        exception_info = None
        proxy_list = [YtProxy(self.Parameters.yt_proxy, yt_token, self.Parameters.fast_features_table)]
        if self.Parameters.use_fallback:
            proxy_list.append(YtFallbackProxy(self.Parameters.fallback_yt_proxy,
                                              yt_token,
                                              self.Parameters.fallback_data_path))
        for proxy in proxy_list:
            logging.info('Using proxy %s', proxy.proxy)
            try:
                self.build(proxy, index_directory)
            except Exception:
                logging.error('Exception occurred on %s: %s', proxy.proxy, traceback.format_exc(sys.exc_info()))
                if not exception_info:
                    exception_info = sys.exc_info()
            else:
                exception_info = None
                break

        if exception_info:
            raise exception_info[0], exception_info[1], exception_info[2]

        if self.Parameters.create_single_resource:
            attrs = {
                'shards_count': str(self.Parameters.shard_count)
            }
            resource = geort.MAPS_DATABASE_FAST_FEATURES_SHARDED(self, self.Parameters.description, index_directory, **attrs)
            sdk2.ResourceData(resource).ready()
        else:
            # It is not allowed for different resources to have common files,
            # so we make copies of mms files for each shard to share them individually.
            files = self._copy_shard_files(self.Parameters.shard_count, index_directory)
            self._share_shards(self.Parameters.shard_count, files)

    def _copy_shard_files(self, shard_count, index_directory):
        '''
        Copy all mms files from indexer output to root task directory (with no hierarchy).
        Retun a list of paths to new files.
        '''
        result = []
        suffixes = list(str(i) for i in range(shard_count))
        if self.Parameters.build_fresh_shard:
            suffixes.append('fresh')
        for i in suffixes:
            fn = 'fast_features.mms.{}'.format(i)
            srcpath = os.path.join(index_directory, '0001', fn)
            if not os.path.isfile(srcpath):
                raise common.errors.TaskFailure('shard #{} not found in indexer output (path: "{}")'.format(i, srcpath))
            os.link(srcpath, fn)  # hard link
            result.append(fn)
        return result

    def _share_shards(self, shard_count, files):
        '''
        Create a single resource for each shard.
        '''
        for i in range(shard_count):
            attrs = {
                'shard_id': str(i),
                'shards_count': str(shard_count),
            }
            resource = geort.MAPS_DATABASE_FAST_FEATURES_SHARD(self, self.Parameters.description, files[i], **attrs)
            sdk2.ResourceData(resource).ready()
        if self.Parameters.build_fresh_shard:
            attrs = {
                'shard_id': 'fresh',
                'shards_count': str(shard_count),
            }
            resource = geort.MAPS_DATABASE_FAST_FEATURES_SHARD(self, self.Parameters.description, files[-1], **attrs)
            sdk2.ResourceData(resource).ready()

    @staticmethod
    def _set_environment(yt_token):
        os.environ['YT_TOKEN'] = yt_token

    def on_release(self, parameters):
        logging.debug("Release parameters: %r", parameters)
        self._send_release_info_to_email(parameters)
        self.mark_released_resources(parameters["release_status"], ttl=7)
