# -*- coding: utf-8 -*-

from sandbox import sdk2
import logging
import tarfile
import requests
import datetime
import time
import sandbox.common.types.task as ctt
from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.geosearch.tools import nirvana
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosearch import resource_types as geo_types
from sandbox.projects.geosearch.ParseAddrsUrlsFromYt import ParseAddrsUrlsFromYt


class BuildMapsWizardOrgNavData(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''
        Build SURL for addrs_wizard
    '''
    class Parameters(sdk2.task.Parameters):
        reference_workflow_id = sdk2.parameters.String(
            'Reference worflow id',
            required=True,
            default_value='2960b79b-061b-4e91-9497-1cea2090f307'
        )
        target_block_name = sdk2.parameters.String('Block with result data',
                                                   required=True,
                                                   default_value='[SURL] Convert index to panther')
        company_table_path = sdk2.parameters.String('Company table path',
                                                    required=True)
        url_parser = sdk2.parameters.Resource('Geosearch YT url parser',
                                              resource_type=geo_types.GEOSEARCH_YT_URL_PARSER,
                                              required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        semaphores = ctt.Semaphores(acquires=[ctt.Semaphores.Acquire(name='BUILD_MAPS_WIZARD_ORG_NAV_DATA')])

        class Caches(sdk2.Requirements.Caches):
            pass

    @sdk2.header()
    def head(self):
        if self.Context.workflow_id:
            url = 'https://nirvana.yandex-team.ru/flow/%s/graph' % self.Context.workflow_id
            return '<a href="{url}">{url}</a>'.format(url=url)
        else:
            return ''

    def make_urls(self):
        if not self.Context.build_business_urls_task:
            parse_urls_task_class = sdk2.Task[ParseAddrsUrlsFromYt.type]
            parse_urls_task = parse_urls_task_class(self,
                                                    description='Build MAPS_WEB_URLS_BUSINESS from YT',
                                                    company_table=self.Parameters.company_table_path,
                                                    url_parser_executable=self.Parameters.url_parser,
                                                    owner=self.owner,
                                                    create_sub_task=True)
            parse_urls_task.enqueue()
            self.Context.build_business_urls_task = parse_urls_task.id
            raise sdk2.WaitTask([self.Context.build_business_urls_task],
                                ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                                wait_all=True)
        else:
            task = sdk2.Task[self.Context.build_business_urls_task]
            if task.status in ctt.Status.Group.BREAK:
                raise SandboxTaskFailureError('Parsing URLs failed')

    def on_execute(self):
        with self.memoize_stage.parse_urls:
            self.make_urls()
        parse_urls_task = sdk2.Task[self.Context.build_business_urls_task]
        business_urls_resource = sdk2.Resource[resource_types.MAPS_WEB_URLS_BUSINESS].find(task=parse_urls_task).first()
        wiki_urls_resource = sdk2.Resource[resource_types.MAPS_WEB_URLS_POI].find(attrs={'released': 'stable'}).first()
        self.result_path = './org_nav_data'
        self.archive_path = './surl.tar.gz'
        token = sdk2.Vault.data('robot-geosearch', 'nirvana_token')
        nirvana_cli = nirvana.Nirvana(token)
        date = datetime.datetime.now().strftime('%Y-%m-%d %H:%M')
        if not self.Context.workflow_id:
            new_workflow = nirvana_cli.clone_workflow(self.Parameters.reference_workflow_id,
                                                      '[SURL] pipeline %s' % date,
                                                      '[SURL] pipeline %s' % date)
            workflow_params = [
                {'parameter': 'timestamp', 'value': int(time.time())},
                {'parameter': 'business_urls_resource_id', 'value': str(business_urls_resource.id)},
                {'parameter': 'poi_urls_resource_id', 'value': str(wiki_urls_resource.id)}
            ]
            nirvana_cli.set_workflow_parameters(new_workflow, workflow_params)
            nirvana_cli.start_workflow(new_workflow)
            self.Context.workflow_id = new_workflow
            logging.info('Waiting workflow %s to complete' % new_workflow)
            raise sdk2.WaitTime(300)
        else:
            while nirvana_cli.get_workflow_state(self.Context.workflow_id)['status'] != 'completed':
                raise sdk2.WaitTime(600)
            if nirvana_cli.get_workflow_state(self.Context.workflow_id)['result'] == 'failure':
                nirvana_msg = nirvana_cli.get_workflow_state(self.Context.workflow_id)['message']
                msg = ('Workflow <a href="https://nirvana.yandex-team.ru/flow/'
                       '{workflow_id}/graph">{workflow_id}</a> failed: '
                       '{workflow_err}')
                raise SandboxTaskFailureError(msg.format(workflow_id=self.Context.workflow_id,
                                                         workflow_err=nirvana_msg))
            else:
                target_block_id = nirvana_cli.get_block_by_name(self.Context.workflow_id,
                                                                self.Parameters.target_block_name)
                storage_path = nirvana_cli.get_target_block_result(self.Context.workflow_id,
                                                                   target_block_id)
                token = sdk2.Vault.data('robot-geosearch', 'nirvana_token')
                download_request = requests.get(storage_path,
                                                verify=False,
                                                headers={'Authorization': 'OAuth {}'.format(token)})
                with open(self.archive_path, 'w') as tar_file:
                    tar_file.write(download_request.content)
                tar_archive = tarfile.open(self.archive_path)
                tar_archive.extractall(path=self.result_path)
                surl_resource = sdk2.Resource[resource_types.MAPS_WIZARD_ORG_NAV_DATA]
                current_surl_resource = surl_resource(self,
                                                      'SURL resource',
                                                      self.result_path)
                surl_data = sdk2.ResourceData(current_surl_resource)
                surl_data.ready()
