import os
import logging

import sandbox.sdk2 as sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.common.types.resource as ctr
import sandbox.common.errors as errors

from sandbox.projects.resource_types import MR_GEMINICL


class CommonMrGeminiParameters(sdk2.Parameters):
    with sdk2.parameters.Group('MR_Geminicl parameters') as gemini_block:
        gemini_resource_id = sdk2.parameters.Resource(
            'MR Geminicl',
            resource_type=MR_GEMINICL,
            state=(ctr.State.READY,),
            required=True,
        )

        mr_gemini_user = sdk2.parameters.String(
            'MR gemini user',
            default='mr-any',
            required=True,
        )
        mr_gemini_canonization_type = sdk2.parameters.String(
            'MR canonization type',
            default='weak',
            required=True,
        )
        mr_gemini_job_count = sdk2.parameters.Integer(
            'MR gemini job count',
            default=1000,
            required=True,
        )
        mr_gemini_job_count_throttle = sdk2.parameters.Integer(
            'MR gemini job count throttle',
            default=300,
            required=True,
        )
        mr_gemini_max_rps = sdk2.parameters.Integer(
            'MR gemini max rps',
            default=50000,
            required=True,
        )


class CanonizeUrls(sdk2.Task):
    """Canonize urls with mr_geminicl"""

    class Parameters(CommonMrGeminiParameters):
        with sdk2.parameters.RadioGroup('Yt host') as yt_host:
            yt_host.values['hahn'] = yt_host.Value(value='Hahn', default=True)
            yt_host.values['banach'] = yt_host.Value(value='Banach')

        with CommonMrGeminiParameters.gemini_block() as gemini_block:
            input_table = sdk2.parameters.String('Yt path to input table', required=True)
            output_table = sdk2.parameters.String('Yt path to output table', required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        with self.memoize_stage.wait_ferryman_batch(commit_on_entrance=False):
            yt_proxy = '{}.yt.yandex.net'.format(self.Parameters.yt_host)
            gemini_runner = self.Parameters.gemini_resource_id

            if gemini_runner is None:
                raise errors.TaskError('No mr_geminicl founded')
            gemini_runner = sdk2.ResourceData(gemini_runner)

            logging.info('Start mr_gemini run')
            with sdk2.helpers.ProcessLog(self, logger='mr_gemini') as pl:
                env = os.environ.copy()
                env['MR_RUNTIME'] = 'YT'
                env['MR_TMP'] = '//tmp/'
                env['YT_TOKEN'] = sdk2.Vault.data(self.owner, 'yt-token')

                run = [
                    str(gemini_runner.path),
                    '--server', yt_proxy,
                    '--user', str(self.Parameters.mr_gemini_user),
                    '--input', self.Parameters.input_table,
                    '--output', self.Parameters.output_table,
                    '--error-output', self.Parameters.output_table + '.err',
                    '--job-count', str(self.Parameters.mr_gemini_job_count),
                    '--job-count-throttle', str(self.Parameters.mr_gemini_job_count_throttle),
                    '--max-rps', str(self.Parameters.mr_gemini_max_rps),
                    '--type', self.Parameters.mr_gemini_canonization_type,
                ]
                ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
                if ret:
                    raise errors.TaskError('mr_gemini run is failed')
