import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
import sandbox.common.types.resource as ctr

from .lib import YaMakeUtils


class CommitFreshGeosearchData(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        dry_run = sdk2.parameters.Bool("Perform all steps but don't commit the result", default=False)
        binary_release = binary_task.binary_release_parameters(stable=True)

    def _reload_with_ttl_inf(self, resource):
        logging.info('Resource #%s has ttl=%s', resource.id, resource.ttl)
        if resource.ttl == 'inf':
            return resource

        # Download the resource
        data = sdk2.ResourceData(resource)

        description = 'Copy of resource #{} with ttl=inf'.format(resource.id)
        new_resource = sdk2.Resource[resource.type](self, description, data.path.name, ttl='inf')

        # To reload the resource with ttl=inf, we have to copy it into the current directory
        # (otherwise, "XXX is not relative to the current task's directory" error is thrown)
        new_data = sdk2.ResourceData(new_resource)
        shutil.copy(data.path, new_data.path)
        new_data.ready()

        logging.info('Resource #%s was reloaded as #%s', resource.id, new_resource.id)
        return new_resource

    def _process_maps_showcase_rubrics(self, old_resource):
        new_resource = sdk2.Resource.find(
            type=old_resource.type,
            attrs={"released": "stable"},
            owner='MAPS_FRONT',
            state=ctr.State.READY,
        ).first()
        if not new_resource:
            raise errors.TaskFailure('Could not find resource')

        if new_resource.id == old_resource.id:
            logging.info('No new resource found: same id %s', new_resource.id)
            return
        if new_resource.md5 is not None and new_resource.md5 == old_resource.md5:
            logging.info('No new resource found: same MD5 hash %s', new_resource.md5)
            return

        new_resource = self._reload_with_ttl_inf(new_resource)
        return new_resource

    def on_execute(self):
        logging.info('Checking out...')
        data_dir = 'data'
        ya_make_file = os.path.join(data_dir, 'ya.make')

        sdk2.svn.Arcadia.checkout('arcadia:/arc/trunk/arcadia/extsearch/geo/meta/rearrs/data', data_dir, depth='empty')
        sdk2.svn.Arcadia.update(ya_make_file)

        updates = []
        for model in YaMakeUtils.get_current_resources(ya_make_file):
            resource = sdk2.Resource[model.resource_id]
            logging.info('Found resource #%s of type %s', resource.id, resource.type)

            if resource.type == 'MAPS_SHOWCASE_RUBRICS':
                new_resource = self._process_maps_showcase_rubrics(resource)
                if new_resource is not None:
                    updates.append((model, new_resource.id))
                    logging.info('Updated #%s to #%s', resource.id, new_resource.id)

        if not updates:
            logging.info('No changes found')
            return

        YaMakeUtils.replace_resources(ya_make_file, ya_make_file, updates)
        with open(ya_make_file) as fp:
            logging.info('Got ya.make:\n%s', fp.read())

        if not self.Parameters.dry_run:
            message = 'Update geosearch data SKIP_CHECK'
            res = sdk2.svn.Arcadia.commit(ya_make_file, message=message, user='zomb-sandbox-rw')
            logging.info('Result: %s', res)
