# -*- coding: utf-8 -*-

import copy
import logging

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sandboxsdk
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.projects.release_machine.input_params as rm_params
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.geo.addrs_meta import GeoMetasearchCfg
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.websearch.middlesearch.CompareMiddlesearchBinaries import CompareMiddlesearchBinaries
from sandbox.projects.websearch.middlesearch.CompareMiddlesearchBinaries import AutoModeParam, LaunchMode
from sandbox.projects.release_machine import rm_notify


def is_redundant(param):
    if hasattr(param, 'group_name'):
        return (
            param.group_name.startswith('Noapacheupper') or
            param.group_name.startswith('Ydo') or
            param.group_name.startswith('models_proxy')
        )
    return param in (AutoModeParam, LaunchMode)


@rm_notify.notify2()
class CompareGeometasearchBinaries(sandboxsdk.task.SandboxTask, object):

    type = 'COMPARE_GEOMETASEARCH_BINARIES'

    environment = [task_env.TaskRequirements.startrek_client]
    client_tags = task_env.TaskTags.startrek_client

    external_parameters = [p for p in CompareMiddlesearchBinaries.input_parameters if not is_redundant(p)]
    input_parameters = [rm_params.ReleaseNum]
    input_parameters.extend(external_parameters)

    def __wait_tasks(self):
        self.wait_tasks(
            self.list_subtasks(load=False),
            self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE,
            wait_all=True,
            state='Waiting for subtasks to complete')

    def __check_tasks(self):
        for task in self.list_subtasks(load=True):
            if not task.is_finished():
                raise SandboxTaskFailureError('Subtask %s has failed (status=%s)' % (task.descr, repr(task.status)))

    def __create_subtask(self, task, input_params, execution_space=None):
        return self.create_subtask(
            task_type=task.type,
            description='%s subtask for #%d (%s)' % (task.type,
                                                     self.id,
                                                     self.descr),
            input_parameters=input_params,
            execution_space=execution_space
        )

    def on_execute(self):
        comparation_modes = ['nocache',
                             'cache',
                             'cachehit']
        self.__check_tasks()
        tasks = {}
        with self.memoize_stage.LAUNCH_TASKS:
            for mode in comparation_modes:
                parameters = copy.deepcopy(self.ctx)
                parameters['auto_mode'] = mode
                parameters['component_name'] = GeoMetasearchCfg.name
                task_id = self.__create_subtask(CompareMiddlesearchBinaries, parameters).id
                tasks.update({mode: task_id})
            self.ctx['comparation_task'] = tasks
            self.__wait_tasks()
        with self.memoize_stage.STARTREK_COMMENTS:
            startrek_token = self.get_vault_data('robot-geosearch',
                                                 'robot_geosearch_startrek_token')
            startrek_helper = STHelper(startrek_token)
            c_info = rmc.COMPONENTS[self.ctx[rm_params.ComponentName.name]]()
            relese_num = self.ctx.get(rm_params.ReleaseNum.name)
            for mode, task_id in self.ctx.get('comparation_task').iteritems():
                task = channel.sandbox.get_task(task_id)
                task_summary = task.ctx.get('summary')
                msg_template = u'Результат работы таска ((https://sandbox.yandex-team.ru/task/{task_id}/view {task_id})) ({mode}):\n{summary}'
                message = msg_template.format(task_id=task_id,
                                              mode=mode,
                                              summary=task_summary)
                message = message.replace("<strong style='color:red'>", '!!(red)')
                message = message.replace('</strong>', '!!')
                logging.info(message)
                startrek_helper.comment(relese_num, message, c_info)
