# -*- coding: utf-8 -*-
import json
import logging

import sandbox.projects.release_machine.core.task_env as task_env
from sandbox import sdk2
from sandbox.projects.geosearch.tools import yappy
from sandbox.projects.geosearch.tools import stoker
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.geosearch.startrek import StartrekClient
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper


class CreateAddrsBeta(sdk2.Task):
    """ Creates yappy beta """

    class Parameters(sdk2.task.Parameters):
        branch = sdk2.parameters.String('Branch',
                                        required=True)
        tag = sdk2.parameters.String('Tag',
                                     required=True)
        resources = sdk2.parameters.List('Resource list',
                                         sdk2.parameters.String)
        with sdk2.parameters.String('Beta type') as beta_type_choices:
            beta_type_choices.values['meta'] = 'meta'
            beta_type_choices.values['wizard'] = 'wizard'
            beta_type_choices.values['base'] = 'base'
            beta_type_choices.values['base_18'] = 'base_18'
            beta_type_choices.values['sprav'] = 'sprav'
            beta_type_choices.values['collections'] = 'collections'
            beta_type_choices.values['geosuggest'] = 'geosuggest'
            beta_type_choices.values['geometasearch'] = 'geometasearch'
        startrek_task = sdk2.parameters.String('Startrek task')
        with sdk2.parameters.String('Launch type') as launch_type_choices:
            launch_type_choices.values['DB'] = 'Database'
            launch_type_choices.values['RM'] = 'Release machine'
        component_name = sdk2.parameters.String('Component name')

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = task_env.TaskTags.startrek_client
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    @sdk2.header()
    def head(self):
        if self.Context.beta_name:
            href = (
                '<a href="https://yappy.z.yandex-team.ru/b/{name}" '
                'target="_blank">{name}</a>'
            ).format(name=self.Context.beta_name)
            return href

    def write_to_startrek(self, comment):
        startrek_token = sdk2.Vault.data('robot-geosearch', 'robot_geosearch_startrek_token')
        if self.Parameters.launch_type_choices == 'RM':
            try:
                startrek_helper = STHelper(startrek_token, useragent='robot-geosearch')
                c_info = rmc.COMPONENTS[self.Parameters.component_name]()
                relese_num = int(self.Parameters.branch)
                startrek_helper.comment(relese_num,
                                        comment,
                                        c_info)
            except Exception:
                logging.info('Could not post comment to startrek')
                logging.info(comment)
        elif self.Parameters.startrek_task and self.Parameters.launch_type_choices == 'DB':
            self.startrek = StartrekClient(startrek_token)
            self.startrek.add_comment(self.Parameters.startrek_task, comment)

    def failed(self):
        msg_template = ('Error creating beta for {branch}-{tag}\n'
                        'Beta creation task: '
                        'https://sandbox.yandex-team.ru/task/{task_id}/view')
        message = msg_template.format(branch=self.Parameters.branch,
                                      tag=self.Parameters.tag,
                                      task_id=self.id)
        self.write_to_startrek(message)

    def _find_upper(self, sources):
        for source in sources:
            if stoker._is_upper(source):
                return source.split(':')

    def set_stoker_config(self):
        token = sdk2.Vault.data('GEOMETA-SEARCH', 'GEOSEARCH_STOKER')
        cgi = None
        if self.Parameters.beta_type_choices == 'meta':
            host, port = self._find_upper(json.loads(self.Context.experimental_sources))
            apphost_port = int(port) + 4
            cgi = [
                'srcrwr=GEOV:{host}:{port}:800'.format(host=host, port=apphost_port),
                'srcrwr=GEOV_SLOW:{host}:{port}:1600'.format(host=host, port=apphost_port),
                'rps-limiter-quota=yappy-global'
            ]
        if self.Parameters.beta_type_choices == 'geometasearch':
            upper_instances = self.Context.instances.get('geometasearch_upper')
            middle_instances = self.Context.instances.get('geometasearch_middle')
            apphost_port = 80 + 4
            cgi = [
                'srcrwr=GEOV:{upper}:{port}:800'.format(upper=upper_instances[0], port=apphost_port),
                'srcrwr=GEOV_SLOW:{upper}:{port}:1600'.format(upper=upper_instances[0], port=apphost_port),
                'srcparams=GEOV:experimental=source=middle:{middle}'.format(middle=middle_instances[0]),
                'srcparams=GEOV_SLOW:experimental=source=middle:{middle}'.format(middle=middle_instances[0]),
                'rps-limiter-quota=yappy-global'
            ]
        if cgi:
            response = stoker.set_bolver_config(self.Context.beta_name, cgi, token)
            logging.info('Stoker config: %s' % response)

    def on_timeout(self, prev_status):
        self.failed()

    def on_failure(self, prev_status):
        self.failed()

    def on_execute(self):
        yappy_token = sdk2.Vault.data('GEOSEARCH_PROD', 'GEOSEARCH_YAPPY_TOKEN')
        if not self.Context.CREATED:
            if not yappy.free_slots(self.Parameters.beta_type_choices, yappy_token):
                msg = ('Can`t find empty slots\n'
                       'Try to restart testenv/sandbox task later')
                self.write_to_startrek(msg)
                raise SandboxTaskFailureError(msg.splitlines()[0])
        nanny_token = sdk2.Vault.data('robot-geosearch', 'ADDRS')
        with self.memoize_stage.CREATE_BETA(commit_on_entrance=False):
            try:
                beta = yappy.YappyBeta(self.Parameters.branch,
                                       self.Parameters.tag,
                                       self.Parameters.beta_type_choices,
                                       self.Parameters.resources,
                                       self.id,
                                       nanny_token,
                                       yappy_token)
                self.Context.beta_name = beta.name
                beta.create()
                self.Context.CREATED = True
                logging.info('Beta name: %s' % beta.name)
                logging.info('Beta patch content: %s' % beta.patch_file)
                logging.info('Beta service: %s' % beta.service)
                logging.info('Beta experimental sources: %s' % beta.experimental_sources)
                logging.info('Beta instances: %s' % beta.instances)
                self.Context.beta_service = beta.service
                self.Context.hamster_url = beta.hamster_url
                self.Context.experimental_sources = json.dumps(beta.experimental_sources)
                self.Context.instances = beta.instances
                self.Context.chaining_dict = {
                    'name': beta.name,
                    'service': beta.service,
                    'hamster_url': beta.hamster_url,
                    'patch': beta.patch_file,
                    'experimental_sources': beta.experimental_sources,
                    'instances': beta.instances
                }
            except Exception as err:
                self.failed()
                raise SandboxTaskFailureError(err)
        resource_path = './beta.json'
        self.set_stoker_config()
        with open(resource_path, 'w') as out:
            json.dump(self.Context.chaining_dict, out)
        service_url = 'https://nanny.yandex-team.ru/ui/#/services/catalog/%s/' % self.Context.beta_service
        msg = ('Beta %s allocated to ((%s %s))\n'
               '<{Beta patch:\n%s}>\n'
               '<{Chaining resource content:\n%s}>') % (self.Context.beta_name,
                                                        service_url,
                                                        self.Context.beta_service,
                                                        '%%{}%%'.format(self.Context.chaining_dict.get('patch')),
                                                        '%%(json){}%%'.format(self.Context.chaining_dict))
        self.write_to_startrek(msg)
