# -*- coding: utf-8 -*-

import os
import logging
import subprocess

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosearch import resource_types as rtypes


class GenerateAddrsTestingBalancerConfig(sdk2.Task):
    """Generate and deploy configuratio for addrs-testing AWACS namespace"""

    class Parameters(sdk2.task.Parameters):
        config_builder = sdk2.parameters.Resource('Config builder binary',
                                                  resource_type=rtypes.ADDRS_TESTING_BALANCER_GENERATOR,
                                                  required=True)
        config_path = sdk2.parameters.ArcadiaUrl('Authorization config Arcadia path',
                                                 default_value='arcadia:/arc/trunk/arcadia/search/geo/tools/production/balancer_configs',
                                                 required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_configs(self):
        return Arcadia.export(self.Parameters.config_path, './balancer_configs')

    def run_generator(self):
        config_builder = sdk2.ResourceData(self.Parameters.config_builder).path
        configs_dir = self.get_configs()
        cmd = (
            '{binary} '
            '--backends_config {configs_dir}/backends.json '
            '--upstreams_config {configs_dir}/upstreams.json '
            '--auth_config {configs_dir}/auth.json'
        ).format(binary=config_builder, configs_dir=configs_dir)
        logging.info('Running: {}'.format(cmd))
        log_file_path = get_logs_folder() + '/balancer_configs_generator.out.txt'
        with open(log_file_path, 'w') as log_file:
            try:
                subprocess.check_call(cmd.split(),
                                      env=self.proc_env,
                                      stdout=log_file,
                                      stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError:
                logging.exception('Building configs failed')
                raise SandboxTaskFailureError('Building namespace failed')

    def on_execute(self):
        self.proc_env = os.environ.copy()
        self.proc_env['AWACS_TOKEN'] = sdk2.Vault.data('GEOMETA-SEARCH', 'awacs-token')
        self.proc_env['NANNY_TOKEN'] = sdk2.Vault.data('robot-geosearch', 'ADDRS')
        self.run_generator()
