# -*- coding: utf-8 -*-
import datetime
import requests

from sandbox import sdk2
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sdk2.helpers import subprocess

from sandbox.projects.common.geosearch.startrek import StartrekClient

from sandbox.projects.geosearch import resource_types as geo_types


def generate_golovan_url(tvm, origin):
    return ("https://yasm.yandex-team.ru/chart/"
            + "itype=rpslimiter;"
            + "hosts=ASEARCH;"
            + "ctype=production;"
            + "prj=main;"
            + ("tag_service={};".format(str(tvm)) if tvm else "")
            + ("tag_user={};".format(origin) if origin else "")
            + "signals=unistat-quotas-default-geo-main-Consumed_dmmm/")


def generate_pushok_comment(approvers, author, ticket_id):
    request = ("https://ok.yandex-team.ru/tracker?"
               + "_embedded=1"
               + "&author={}".format(author)
               + "&object_id={}".format(ticket_id)
               + "&uid={}".format(datetime.datetime.now().isoformat())
               + "&text=Согласны ли с выставленной квотой?"
               + "&is_parallel=true")
    for login in approvers:
        request += "&user={}".format(login)

    return ("{{iframe src=\""
            + request
            + "\" frameborder=0 width=100% height=400px scrolling=no}}"
            + "\nСервис для согласований ((https://wiki.yandex-team.ru/intranet/ok/ OK)). "
            + "((https://st.yandex-team.ru/createTicket?"
            + "summary=%5BФидбек%5D&type=2&priority=2&tags=feedback&queue=OK Обратная связь)).")


class GenerateGeosearchQuotaTicket(sdk2.Task):
    """
        Generate startrack ticket in  queue GEOSEARCHQUOTA for new or current client
    """

    class Parameters(sdk2.Task.Parameters):
        tvm = sdk2.parameters.Integer("TVM client_id")
        origin = sdk2.parameters.String("origin=... parameter in requests from client (only if it doesn't use TVM)")
        rps = sdk2.parameters.Integer("Requseted RPS in quota")
        rps_coef = sdk2.parameters.Float("Coef which is multiplied by 95%-quantile of client RPS")
        approvers = sdk2.parameters.String("Comma separated list of logins")

        rps_stat_calcer = sdk2.parameters.Resource(
            'Tool for printing report about RPS from client',
            resource_type=geo_types.GEOSEARCH_RPS_STAT_CALCER,
            required=True
        )

    def on_execute(self):
        executable = str(sdk2.ResourceData(self.Parameters.rps_stat_calcer).path)

        if not self.Parameters.tvm and not self.Parameters.origin:
            raise SandboxTaskFailureError('TVM client_id or origin must be provided')

        if not self.Parameters.rps and not self.Parameters.rps_coef:
            raise SandboxTaskFailureError('Quota RPS or coef for calculating it must be provided!')

        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        startrek_client = StartrekClient(startrek_token)

        approvers = []
        if self.Parameters.approvers:
            approvers = list(str(self.Parameters.approvers).split(','))
        followers = ["yu-semichev", "evelkin"]
        followers.extend(approvers)

        cmd = [
            executable,
            "--output", "rps_report.txt",
        ]
        if self.Parameters.tvm:
            cmd.extend(["--tvm", str(self.Parameters.tvm)])
            "--tvm", str(self.Parameters.tvm)
        else:
            cmd.extend(["--origin", str(self.Parameters.origin)])
        subprocess.check_call(cmd)

        rps_report = open("rps_report.txt", 'r').read()
        unknown_client = "No data" in rps_report
        if unknown_client and self.Parameters.rps_coef:
            raise SandboxTaskFailureError('Using RPS coef for new clients is impossible!')

        tvm_desc = ""
        q95_rps = 0.
        max_rps = 0.
        for line in rps_report.split('\n'):
            if line.startswith("TVM: "):
                tvm_desc = line[line.find('(') + 1:line.find(')')]
            elif line.startswith("Q95: "):
                q95_rps = float(line[len("Q95: "):])
            elif line.startswith("Max: "):
                max_rps = float(line[len("Max: "):line.find('(')])

        if self.Parameters.tvm and not tvm_desc:
            tvm_api_resp = requests.get("https://tvm.yandex-team.ru/client/"
                                        + str(self.Parameters.tvm)
                                        + "/info").json()
            tvm_desc = tvm_api_resp["name"]

        ticket_name = "Квота на походы в геопоиск "
        if tvm_desc:
            ticket_name += "для клиента c TVM {tvm} ({desc})".format(
                tvm=self.Parameters.tvm,
                desc=tvm_desc
            )
        else:
            ticket_name += "с origin=" + str(self.Parameters.origin)

        rps = int(self.Parameters.rps) if self.Parameters.rps else int(q95_rps * float(self.Parameters.rps_coef))
        ticket_description = "Предлагается квота в {} RPS\n".format(rps)
        ticket_description += "График RPS:\n"
        ticket_description += generate_golovan_url(int(self.Parameters.tvm), str(self.Parameters.origin)) + "\n\n"
        if not unknown_client:
            ticket_description += "\nСтатистика по текущему RPS:\n" + rps_report + "\n"
        if max_rps > rps:
            ticket_description += "ВНИМАНИЕ: максимальное RPS за неделю превышает предлагаемый размер квоты!\n"

        ticket_dict = {'queue': 'GEOSEARCHQUOTA',
                       'summary': ticket_name,
                       'description': ticket_description}
        ticket_id = startrek_client.create_task(ticket_dict)
        startrek_client.add_followers(ticket_id, followers)
        startrek_client.add_comment(ticket_id, generate_pushok_comment(approvers, "yu-semichev", ticket_id))
