# -*- coding: utf-8 -*
import os
import logging
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects import resource_types as rtypes
from sandbox.projects.geosearch import resource_types as geotypes
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.nanny import nanny


class ParseAddrsUrlsFromYt(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''
        Downloads URLs from YT company table
    '''

    class Parameters(sdk2.task.Parameters):
        company_table = sdk2.parameters.String('YT company table',
                                               required=True)
        url_parser_executable = sdk2.parameters.Resource('Url parser executable',
                                                         resource_type=geotypes.GEOSEARCH_YT_URL_PARSER)
        data_preparer_executable = sdk2.parameters.Resource('Geosearch YT data preparer',
                                                            resource_type=geotypes.GEOSEARCH_YT_DATA_PREPARER)
        shards_count = sdk2.parameters.Integer('Number of shards')

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_release(self, params):
        sdk2.Task.on_release(self, params)
        # {u'release_comments': u'base builded from 1st october export',
        # u'release_subject': u'1st of october', u'release_status': u'testing',
        # u'email_notifications': {u'cc': [], u'to': []}, u'releaser':
        # u'karas-pv'}
        release_dict = dict(releaser=params['releaser'],
                            release_status=params['release_status'],
                            release_subject=params['release_subject'],
                            email_notifications=dict(to=['karas-pv@yandex-team.ru'], cc=[]),
                            release_comments=params['release_comments'])
        nanny.ReleaseToNannyTask2.on_release(self, release_dict)

    def on_execute(self):
        proc_env = os.environ.copy()
        yt_token = sdk2.Vault.data('GEOMETA-SEARCH', 'yt-token')
        proc_env['YT_TOKEN'] = yt_token
        output_file = './urls.txt'
        parser_executable = sdk2.ResourceData(self.Parameters.data_preparer_executable or self.Parameters.url_parser_executable).path
        cmd = ('{binary} {mode} '
               '-s hahn '
               '-i {company_table_path} '
               '-o {output_file_path}').format(binary=parser_executable,
                                               mode=('urls' if self.Parameters.data_preparer_executable else ''),
                                               company_table_path=self.Parameters.company_table,
                                               output_file_path=output_file)
        if self.Parameters.data_preparer_executable and self.Parameters.shards_count and self.Parameters.shards_count > 0:
            cmd += (' --shard-count {cnt}'.format(cnt=self.Parameters.shards_count))
        logging.info('Running: %s' % cmd)
        log_file_path = get_logs_folder() + '/parse_urls_from_export_yt.out.txt'
        with open(log_file_path, 'w') as log_file:
            try:
                subprocess.Popen(cmd,
                                 env=proc_env,
                                 shell=True,
                                 stdout=log_file,
                                 stderr=subprocess.STDOUT).wait()
            except subprocess.CalledProcessError as err:
                logging.info('%s command failed' % cmd)
                logging.info('Details %s:\n' % err)
                raise SandboxTaskFailureError('Getting URLs failed')
        business_urls_resource = sdk2.Resource[rtypes.MAPS_WEB_URLS_BUSINESS]
        current_business_urls_resource = business_urls_resource(self,
                                                                'Made from %s' % self.Parameters.company_table,
                                                                output_file)
        business_urls_data = sdk2.ResourceData(current_business_urls_resource)
        business_urls_data.ready()
