# -*- coding: utf-8 -*-

import os
import logging
import subprocess
from datetime import datetime

from sandbox import sdk2
from sandbox.projects import resource_types
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import get_logs_folder

from sandbox.projects.geosearch.tools.nanny import Nanny
from sandbox.projects.geosearch.tools.stoker import _is_upper
from sandbox.projects.geosearch.tools.metasearch_log_parser import parse_eventlog_file


class ParseGeometasearchLogs(sdk2.Task):

    class Parameters(sdk2.Parameters):
        service = sdk2.parameters.String('Beta service name')
        start_time = sdk2.parameters.String('Dump logs from')
        end_time = sdk2.parameters.String('Dump logs till')

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_hosts(self):
        instances = []
        for data in self.nanny_client.get_isolated_instances(self.Parameters.service):
            host = data.get('container_hostname')
            port = data.get('port')
            instances.append({
                'hostname': host,
                'port': port,
                'type': 'upper' if _is_upper('{}:{}'.format(host, port)) else 'middle'
            })
        return instances

    def download_log(self, host_data, hostname_suffix=''):
        with sdk2.ssh.Key(self, "GEOMETA-SEARCH", "robot-geosearch-ssh"):
            configuration_id = self.nanny_client.get_configuration_id(self.Parameters.service)
            ssh_auth = '"ssh -l  \'//user:robot-geosearch//configuration_id:{service}#{conf_id}\'"'.format(
                service=self.Parameters.service,
                conf_id=configuration_id
            )
            files = {
                'eventlog_file_name': 'current-eventlog-addrs{service}-{port}'.format(
                    service=host_data.get('type'),
                    port=host_data.get('port')
                ),
                'stderr_log_name': 'meta.err'
            }
            for file_type, remote_file_name in files.iteritems():
                local_file_name = '{host}_{f_name}{suffix}'.format(
                    host=host_data.get('hostname'),
                    f_name=remote_file_name.replace('current-', ''),
                    suffix=hostname_suffix
                )
                log_directory = '/logs/' if file_type == 'eventlog_file_name' else ''
                cmd = 'rsync -av --copy-links -e {auth} {host}:{directory}{remote_name} {local_name}'.format(
                    auth=ssh_auth,
                    host=host_data.get('hostname'),
                    directory=log_directory,
                    remote_name=remote_file_name,
                    local_name=local_file_name
                )
                log_path = os.path.join(get_logs_folder(), remote_file_name + '_download.log')
                logging.info('Rsync command: {}'.format(cmd))
                with open(log_path, 'w') as log:
                    try:
                        subprocess.call(
                            cmd,
                            shell=True,
                            stderr=subprocess.STDOUT,
                            stdout=log)
                        if not os.path.exists(local_file_name):
                            logging.info('Failed to download {} file'.format(remote_file_name))
                        else:
                            logging.info(
                                'File {remote_name} downloaded from {host} as {local_name}'.format(
                                    remote_name=remote_file_name,
                                    host=host_data.get('hostname'),
                                    local_name=local_file_name
                                )
                            )
                            files.update({file_type: local_file_name})
                    except subprocess.CalledProcessError:
                        raise SandboxTaskFailureError('Failed to download logfiles from {host}'.format(host=host_data.get('hostname')))
            return files

    def download_logs(self, host_data):
        for data in host_data:
            downloaded_files = self.download_log(data)
            data.update(downloaded_files)

    def get_evlogdump_binary(self):
        latest_evlogdump = sdk2.Resource[resource_types.EVLOGDUMP_EXECUTABLE].find(attrs={'released': 'stable'}).first()
        return str(sdk2.ResourceData(latest_evlogdump).path)

    def convert_time(self, time_str):
        d = datetime.strptime(time_str, '%d.%m.%YT%H:%M')
        return d.strftime('%Y-%m-%dT%H:%M')

    def dump_eventlog(self, data, evlogdump_binary):
        output_file = '{}_dumped'.format(data.get('eventlog_file_name'))
        cmd = '{binary} -s {start} -e {end} {evlog_file} > {output_file}'.format(
            binary=evlogdump_binary,
            start=self.convert_time(self.Parameters.start_time),
            end=self.convert_time(self.Parameters.end_time),
            evlog_file=data.get('eventlog_file_name'),
            output_file=output_file
        )
        log_path = os.path.join(get_logs_folder(), data.get('eventlog_file_name') + '_dump.log')
        with open(log_path, 'w') as log:
            try:
                subprocess.call(
                    cmd,
                    shell=True,
                    stderr=subprocess.STDOUT,
                    stdout=log
                )
                if not os.path.exists(output_file):
                    logging.info('Failed to dump {} file'.format(data.get('eventlog_file_name')))
                else:
                    plain_text_resource = sdk2.Resource[resource_types.PLAIN_TEXT](
                        self,
                        output_file,
                        output_file
                    )
                    plain_text_data = sdk2.ResourceData(plain_text_resource)
                    plain_text_data.ready()
                    return output_file
            except subprocess.CalledProcessError:
                raise SandboxTaskFailureError('Failed to dump eventlog')

    def dump_eventlogs(self, host_data):
        evlogdump_binary = self.get_evlogdump_binary()
        for data in host_data:
            dumped_logs = self.dump_eventlog(data, evlogdump_binary)
            data.update({'dumped_log_file_name': dumped_logs})

    def parse_eventlog(self, file_path):
        frames = parse_eventlog_file(file_path)
        return frames

    def parse_eventlogs(self, host_data):
        for data in host_data:
            parsed_frames = self.parse_eventlog(data.get('dumped_log_file_name'))
            data.update({'eventlog_erros': parsed_frames})

    def parse_stderr(self, file_path):
        pass

    def on_execute(self):
        self.nanny_client = Nanny(sdk2.Vault.data('robot-geosearch', 'ADDRS'))
        hosts = self.get_hosts()
        self.download_logs(hosts)
        self.dump_eventlogs(hosts)
        self.parse_eventlogs(hosts)
        self.Context.result = hosts
