import time

import sandbox.sdk2 as sdk2
import sandbox.sandboxsdk.environments as environments

from sandbox.projects.common import solomon


class CommonSolomonParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Solomon parameters') as solomon_block:
        solomon_project = sdk2.parameters.String('Project', required=True)
        solomon_cluster = sdk2.parameters.String('Cluster', required=True)
        solomon_service = sdk2.parameters.String('Service', required=True)


class PushToSolomonYtTablesSizes(sdk2.Task):
    """Send sensors via solomon PUSH-api"""

    class Parameters(CommonSolomonParameters):
        with sdk2.parameters.Group('Yt parameters') as yt_block:
            with sdk2.parameters.RadioGroup('Yt host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(value='Hahn', default=True)
                yt_host.values['banach'] = yt_host.Value(value='Banach')

            path = sdk2.parameters.String('Table path', required=True)

            send_row_count = sdk2.parameters.Bool('Send row count?')
            with send_row_count.value[True]:
                row_count_sensor = sdk2.parameters.String('Solomon sensor for row count')

            send_data_size = sdk2.parameters.Bool('Send data size?')
            with send_data_size.value[True]:
                data_size_sensor = sdk2.parameters.String('Solomon sensor for data_size')

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        commonLabels = {
            'project': self.Parameters.solomon_project,
            'cluster': self.Parameters.solomon_cluster,
            'service': self.Parameters.solomon_service,
        }

        sensors = list()
        ts = int(time.time())

        import yt.wrapper as yt

        yt.config['token'] = sdk2.Vault.data(self.owner, 'yt-token')
        yt.config['proxy']['url'] = '{}.yt.yandex.net'.format(self.Parameters.yt_host)
        if self.Parameters.send_row_count:
            sensors.append(
                {
                    'labels': {'sensor': self.Parameters.row_count_sensor, },
                    'ts': ts,
                    'value': int(yt.row_count(self.Parameters.path)),
                }
            )

        if self.Parameters.send_data_size:
            sensors.append(
                {
                    'labels': {'sensor': self.Parameters.data_size_sensor, },
                    'ts': ts,
                    'value': int(yt.get_attribute(self.Parameters.path, 'uncompressed_data_size')),
                }
            )

        if sensors:
            solomon.upload_to_solomon(commonLabels, sensors)
