# -*- coding: utf-8 -*
import os
import re
import json
import logging
import subprocess


from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects import resource_types as rtypes
from sandbox.projects.geosearch import resource_types as geotypes
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.BuildMapsStaticFactorsDownloader import MapsRubricStaticFactors
from sandbox.projects.common.nanny import nanny


class RegisterAddrsBaseShard(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''
        Downloads shard from YT and register in on shardtracker
    '''

    class Parameters(sdk2.task.Parameters):
        index_table = sdk2.parameters.String('YT index table',
                                             required=True)
        shard_number = sdk2.parameters.Integer('Shard number',
                                               required=True)
        companies_mtime = sdk2.parameters.Integer('Companies table modification time',
                                                  required=False)
        cooking_start_time = sdk2.parameters.Integer('Index cooking start time',
                                                     required=False)
        company_table_path = sdk2.parameters.String('YT path to company table',
                                                    required=False)
        filtering_annotations_table_path = sdk2.parameters.String('YT path to filtering annotations table',
                                                                  required=False)
        geo_stat = sdk2.parameters.Resource('Geo.stat',
                                            resource_type=rtypes.GEO_STAT)
        annotations = sdk2.parameters.Resource('Annotations',
                                               resource_type=rtypes.MAPS_WEB_INDEXANN)
        rubric_static_factors_yt = sdk2.parameters.Resource('Rubric static factors from YT',
                                                            resource_type=MapsRubricStaticFactors)
        docurl_static_factors_yt = sdk2.parameters.Resource('Docurl static factors from YT',
                                                            resource_type=geotypes.GEOSEARCH_STATIC_FACTPRS_DOCURL)

        precomputed_filters = sdk2.parameters.Resource('File with precomputed filters',
                                                            resource_type=rtypes.BUSINESS_COMPUTED_FILTERS)
        features = sdk2.parameters.Resource('File with features',
                                            resource_type=geotypes.GEOSEARCH_FEATURES)

        index_downloader_executable = sdk2.parameters.Resource('Index downloader executable',
                                                               resource_type=geotypes.GEOSEARCH_INDEX_DOWNLOADER)

        base_commit_id = sdk2.parameters.Integer('Commit id of sprav base cooking',
                                                 required=False)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        client_tags = ctc.Tag.GENERIC & (ctc.Tag.MAN | ctc.Tag.SAS | ctc.Tag.VLA)

        class Caches(sdk2.Requirements.Caches):
            pass

    def download_shard(self):
        proc_env = os.environ.copy()
        yt_token = sdk2.Vault.data('GEOMETA-SEARCH', 'yt-token')
        proc_env['YT_TOKEN'] = yt_token
        downloader = sdk2.ResourceData(self.Parameters.index_downloader_executable).path
        shard_path = './shard'
        cmd = ('{binary} '
               '-s hahn '
               '-i {index_table} '
               '-n {shard_num} '
               '-d {shard_path}').format(binary=downloader,
                                         index_table=self.Parameters.index_table,
                                         shard_num=self.Parameters.shard_number,
                                         shard_path=shard_path)
        logging.info('Running: %s' % cmd)
        log_file_path = get_logs_folder() + '/download.out.txt'
        with open(log_file_path, 'w') as log_file:
            try:
                res_path = './index_0000000000_0000000000'
                subprocess.check_call(cmd,
                                      shell=True,
                                      env=proc_env,
                                      stdout=log_file,
                                      stderr=subprocess.STDOUT)
                os.renames(shard_path + '/index_0000000000_0000000000',
                           res_path)
                # hack for OSM build
                for f in ['chains.pbs', 'features.pbs', 'indexfactorann.key']:
                    if not os.path.exists(os.path.join(res_path, f)):
                        open(os.path.join(res_path, f), 'a').close()
                return res_path
            except subprocess.CalledProcessError as err:
                logging.info('%s command failed' % cmd)
                logging.info('Details %s:\n' % err)
                raise SandboxTaskFailureError('Downloading shard failed')

    def make_shard_info(self):
        if self.Parameters.companies_mtime and self.Parameters.company_table_path:
            info = {'resource_timestamp': self.Parameters.companies_mtime,
                    'company_table_path': self.Parameters.company_table_path,
                    'shard_id': self.Parameters.shard_number}
            if self.Parameters.filtering_annotations_table_path:
                info['filtering_annotations_table_path'] = self.Parameters.filtering_annotations_table_path
            if self.Parameters.cooking_start_time:
                info['cooking_start_time'] = self.Parameters.cooking_start_time
            if self.Parameters.base_commit_id:
                info['base_commit_id'] = self.Parameters.base_commit_id
            info_path = os.path.join(self.shard_path, 'shard_info.json')
            with open(info_path, 'w') as out:
                json.dump(info, out)
            logging.info('Shard info written to %s' % info_path)

    def copy_additional_resources(self):
        annotation_path = str(sdk2.ResourceData(self.Parameters.annotations).path)
        geo_stat_path = str(sdk2.ResourceData(self.Parameters.geo_stat).path)
        rubric_static_factors_path = str(sdk2.ResourceData(self.Parameters.rubric_static_factors_yt).path)
        docurl_static_factors_path = str(sdk2.ResourceData(self.Parameters.docurl_static_factors_yt).path)
        precomputed_filters_path = str(sdk2.ResourceData(self.Parameters.precomputed_filters).path)
        files = [os.path.join(annotation_path, file_name) for file_name in os.listdir(annotation_path)]
        files.append(rubric_static_factors_path)
        files.append(docurl_static_factors_path)
        files.append(precomputed_filters_path)
        if self.Parameters.features:
            files.append(str(sdk2.ResourceData(self.Parameters.features).path))
        for path in files:
            logging.info('Trying to copy %s to %s' % (path, self.shard_path))
            new_path = os.path.join(self.shard_path, os.path.basename(path))
            copy_path(path, new_path)
        # God bless geo.stat!
        new_geo_stat_path = os.path.join(self.shard_path, 'geo.stat')
        logging.info('Trying to copy %s to %s' % (geo_stat_path,
                                                  self.shard_path))
        copy_path(geo_stat_path, new_geo_stat_path)

    def patch_shard_conf(self):
        shard_conf_path = self.shard_path + '/shard.conf'
        with open(shard_conf_path) as shard_conf_file:
            shard_conf = shard_conf_file.read()
        with open(shard_conf_path, 'w') as shard_conf_file:
            rgx = re.compile(r'(?:00)?\$\{SHARD_ID\}')
            replace_pattern = rgx.findall(shard_conf)[0]
            shard_number = '{number:0>3d}'.format(number=self.Parameters.shard_number)
            shard_conf = shard_conf.replace(replace_pattern, shard_number)
            shard_conf = shard_conf.replace('${SANDBOX_TASK_ID}', str(self.id))
            shard_conf_file.write(shard_conf)
        logging.info('Patched %s file' % shard_conf_path)

    def get_shard_name(self):
        shard_conf_path = self.shard_path + '/shard.conf'
        with open(shard_conf_path) as f:
            shard_conf = f.read()
        rgx = re.compile(r'%shard\s(addrs_base_prod-0.*)\n%mtime')
        shard_name = rgx.findall(shard_conf)[0]
        return shard_name

    def create_shard_resourse(self):
        shard_resource = sdk2.Resource[rtypes.ADDRS_BUSINESS_SHARD]
        current_shard_resource = shard_resource(self,
                                                'Shard for addrs base search',
                                                self.shard_path)
        current_shard_resource.backup_task = True
        current_shard_resource.shard_name = self.get_shard_name()
        if self.Parameters.shard_number == 0 and self.parent.Parameters.need_acceptance:
            current_shard_resource.addrs_base_shard_for_testenv = True
        shard_data = sdk2.ResourceData(current_shard_resource)
        shard_data.ready()

    def on_execute(self):
        self.shard_path = self.download_shard()
        self.make_shard_info()
        self.copy_additional_resources()
        self.patch_shard_conf()
        self.create_shard_resourse()
        self.Context.shard_name = self.get_shard_name()
