# -*- coding: utf-8 -*-
from sandbox import sandboxsdk
from sandbox import common

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosearch.ReleaseAddrsShardmap import ReleaseAddrsShardmap
from sandbox.projects.AddrsDeploy.task import AddrsDeploy
from sandbox.projects.AddrsDeploy.parameters import get_nanny_cache


class Basesearch(sandboxsdk.parameters.SandboxStringParameter):
    name = 'basesearch_service'
    description = 'Basesearch service'
    default_value = 'addrs_base'
    required = True

    @common.utils.singleton_classproperty
    def choices(cls):
        data = get_nanny_cache()
        services = sorted(data.get('services'))
        return [(service, service) for service in services if 'base' in service]


class Wizard(sandboxsdk.parameters.SandboxStringParameter):
    name = 'wizard_service'
    description = 'Wizard service'
    default_value = 'addrs_wizard'
    required = True

    @common.utils.singleton_classproperty
    def choices(cls):
        data = get_nanny_cache()
        services = sorted(data.get('services'))
        return [(service, service) for service in services if 'wizard' in service]


# class ReleaseStatus(sandboxsdk.parameters.SandboxStringParameter):
#     name = 'release_status'
#     description = 'Release status for resources'
#     default_value = 'stable'
#     #required = True
#     choices = [('stable', 'stable'),
#                ('testing', 'testing')]


class ShardmapId(sandboxsdk.parameters.ResourceSelector):
    name = 'shardmap_resource_id'
    description = 'Shardmap'
    required = True
    resource_type = resource_types.ADDRS_BUSINESS_SHARDMAP


class ResourceIds(sandboxsdk.parameters.ListRepeater, sandboxsdk.parameters.SandboxStringParameter):
    name = 'resource_id'
    description = 'Resources to release'


class ReleaseAddrsShardAndAdditionalResources(sandboxsdk.task.SandboxTask):

    type = 'RELEASE_ADDRS_SHARD_AND_ADDITIONAL_RESOURCES'

    input_parameters = [Basesearch,
                        Wizard,
                        # ReleaseStatus,
                        ShardmapId,
                        ResourceIds]

    def get_task_for_release(self, resource_id):
        resource = channel.sandbox.get_resource(resource_id)
        task_id = resource.task_id
        return task_id

    def __wait_tasks(self):
        self.wait_tasks(
            self.list_subtasks(load=False),
            self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE,
            wait_all=True,
            state='Waiting for subtasks to complete')

    def __check_tasks(self):
        for task in self.list_subtasks(load=True):
            if not task.is_finished():
                msg = 'Subtask %s has failed (status=%s)' % (task.descr,
                                                             repr(task.status))
                raise SandboxTaskFailureError(msg)

    def __create_subtask(self, task, input_params, execution_space=None):
        return self.create_subtask(
            task_type=task.type,
            description='%s subtask for #%d (%s)' % (task.type,
                                                     self.id,
                                                     self.descr),
            input_parameters=input_params,
            execution_space=execution_space
        )

    def on_execute(self):
        self.basesearch = self.ctx.get(Basesearch.name)
        self.wizard = self.ctx.get(Wizard.name)
        # self.release_status = self.ctx.get(ReleaseStatus.name)
        self.shardmap_resource_id = self.ctx.get(ShardmapId.name)
        self.resource_ids = [str(resource) for resource in self.ctx.get(ResourceIds.name)]
        # Always check children
        self.__check_tasks()
        with self.memoize_stage.RELEASE_ADDRS_SHARDMAP:
            release_shardmap_parameters = {'service': self.basesearch,
                                           'shardmap': self.shardmap_resource_id}
            self.release_shardmap_task = self.__create_subtask(ReleaseAddrsShardmap,
                                                               release_shardmap_parameters)
            self.ctx['release_shardmap_task'] = self.release_shardmap_task.id
            deploy_surl_parameters = {'action': 'sandbox_resource',
                                      'activate': True,
                                      'destination_service': self.wizard,
                                      'nanny_resource': self.resource_ids}
            self.deploy_surl_task = self.__create_subtask(AddrsDeploy,
                                                          deploy_surl_parameters)
            self.ctx['deploy_surl_task'] = self.deploy_surl_task.id
            self.__wait_tasks()


__Task__ = ReleaseAddrsShardAndAdditionalResources
