import time
import json
import logging
import requests

import sandbox.sdk2 as sdk2

import sandbox.common.errors as errors


class CommonFerrymanParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Ferryman parameters') as ferryman_block:
        ferryman_pen = sdk2.parameters.String('Your ferryman pen', required=True)
        namespace = sdk2.parameters.String('Namespace', required=True)
        is_delta = sdk2.parameters.Bool('Delta', default=False)
        with sdk2.parameters.RadioGroup('Cluster') as yt_cluster:
            yt_cluster.values[''] = yt_cluster.Value(value='None', default=True)
            yt_cluster.values['hahn'] = yt_cluster.Value(value='Hahn')
            yt_cluster.values['banach'] = yt_cluster.Value(value='Banach')
            yt_cluster.values['arnold'] = yt_cluster.Value(value='Arnold')


class SendTablesToFerryman(sdk2.Task):
    """Send request to HTTP-pen"""

    class Parameters(CommonFerrymanParameters):
        with CommonFerrymanParameters.ferryman_block() as ferryman_block:
            table = sdk2.parameters.String('Yt path to table', required=True)
            timestamp = sdk2.parameters.Integer('Timestamp', default=None)

            wait = sdk2.parameters.Bool('Wait processing?', default=False)

            with wait.value[True]:
                wait_period = sdk2.parameters.Integer('Minutes between batch-status', default=5, required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def send_ferryman_request(self):
        if self.Parameters.timestamp is not None:
            timestamp = self.Parameters.timestamp
        else:
            timestamp = int(time.time() * 10 ** 6)

        common_json = {
            'Path': self.Parameters.table,
            'Namespace': self.Parameters.namespace,
            'Timestamp': timestamp,
            'Delta': self.Parameters.is_delta
        }
        if self.Parameters.yt_cluster:
            common_json['Cluster'] = self.Parameters.yt_cluster
        query = {
            'tables': json.dumps(
                [
                    common_json
                ],
                separators=(',', ':')
            ),
        }
        response = requests.get('{}/add-full-tables'.format(self.Parameters.ferryman_pen), query)
        logging.info(query['tables'])
        logging.info(response.url)
        logging.info(response.text)
        try:
            self.Context.batch = response.json()['batch']
        except:
            raise errors.TaskError('Request to Ferryman is failed')

    def wait_ferryman_batch(self):
        if not self.Parameters.wait:
            return
        query = {
            'batch': self.Context.batch
        }

        response = requests.get('{}/get-batch-status'.format(self.Parameters.ferryman_pen), query)
        logging.info(response.text)
        try:
            response_json = response.json()
        except:
            raise errors.TaskError('Get batch status is failed')

        if response_json['status'] in ('queue', 'processing', 'transfer'):
            raise sdk2.WaitTime(self.Parameters.wait_period * 60)
        elif (response_json['status'] == 'error'
              or (response_json['status'] == 'final' and response_json.get('invalid_input', []))):
            raise errors.TaskFailure('Error occurred in batch processing')
        elif response_json['status'] == 'final':
            return
        else:
            raise errors.TaskFailure('Can\'t parse answer')

    def on_execute(self):
        with self.memoize_stage.send_ferryman_request:
            self.send_ferryman_request()

        with self.memoize_stage.wait_ferryman_batch(commit_on_entrance=False, commit_on_wait=False):
            self.wait_ferryman_batch()
