import datetime
import logging

from sandbox import sdk2
from sandbox.sandboxsdk import environments
import sandbox.projects.geosearch.tools.fast_feature as ff


class SyncFastFeatureFallbackData(sdk2.Task):
    """ Fast feature fallback data synchronization """

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 600

        # custom parameters
        yt_token = sdk2.parameters.YavSecret('YT token', default='sec-01daxf0w5f7c1av3p04jazan1e')
        source_proxy = sdk2.parameters.String('Source cluster', default='hahn', required=True)
        source_list = sdk2.parameters.List('Source path list', default=[
            '//home/apteki/export/geosearch_export',
            '//home/apteki/export/geosearch_export_zdravcity',
            '//home/afisha/export/production/schedules',
            '//home/sprav/fast_features/stable/pharmacy_on_duty_turkey_partner',
            '//home/sprav/altay/prod/export/source/fast_feature_providers',
            '//home/sprav/fast_export/fresh_states',
            '//home/travel/prod/prices/price_filter_data/latest/price_filter_data',
            '//home/sprav/quarantine/fixlist_rubrics',
            '//home/sprav/quarantine/fixlist_chains',
        ])
        fallback_proxy = sdk2.parameters.String('Fallback cluster', default='arnold', required=True)
        fallback_data_dir = sdk2.parameters.String('Fallback data directory',
                                                   default='//home/geo-search/fast-feature-fallback',
                                                   required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1 << 10
        ram = 1 << 10
        environments = [
            environments.PipEnvironment('yandex-yt', version='0.10.8'),
            environments.PipEnvironment('yandex-yt-transfer-manager-client'),
        ]

        class Caches(sdk2.Task.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def on_execute(self):
        import yt.wrapper as yt
        import yt.transfer_manager.client.client as tm
        from yt.common import datetime_to_string

        token = self.Parameters.yt_token.data()['secret']

        yt.config['token'] = token
        source_client = yt.YtClient(proxy=self.Parameters.source_proxy, config=yt.config.config)
        fallback_client = yt.YtClient(proxy=self.Parameters.fallback_proxy, config=yt.config.config)

        with source_client.Transaction():
            source_temp_dir = source_client.find_free_subpath('//tmp/')
            source_client.create('map_node', source_temp_dir)
            expiration_time = datetime.datetime.utcnow() + datetime.timedelta(seconds=self.Parameters.kill_timeout)
            expiration_time = datetime_to_string(expiration_time)
            source_client.set(yt.ypath_join(source_temp_dir, '@expiration_time'), expiration_time)
        source_client.set(yt.ypath_join(source_temp_dir, '@account'), 'geosearch')

        to_copy = []
        with source_client.Transaction():
            for source_path in self.Parameters.source_list:
                source_client.lock(source_path, mode='snapshot')
                source_time = ff.get_source_time(source_client, source_path)
                logging.debug('Source time on %s: %s (%s)', self.Parameters.source_proxy, source_time,
                              source_path)

                fallback_path, fallback_time = ff.fallback_source(fallback_client,
                                                                  self.Parameters.fallback_data_dir,
                                                                  source_path)

                if source_time and (not fallback_time or fallback_time < source_time):
                    logging.info('Transfer source from %s to %s (%s)', self.Parameters.source_proxy,
                                 self.Parameters.fallback_proxy, source_path)
                    temp = source_client.create_temp_table(source_temp_dir)
                    source_client.run_merge(source_path, temp)
                    source_time = datetime_to_string(source_time)
                    source_client.set(yt.ypath_join(temp, '@source_time'), source_time)
                    to_copy.append([temp, ff.fallback_path(self.Parameters.fallback_data_dir, source_path)])

        transfer = tm.TransferManager(token=token)
        transfer.add_tasks_from_src_dst_pairs(
            source_cluster=self.Parameters.source_proxy,
            destination_cluster=self.Parameters.fallback_proxy,
            src_dst_pairs=to_copy,
            sync=True,
        )
