import json
from datetime import datetime
import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.resource_types.releasers as resource_releasers
from sandbox.projects import resource_types
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr

from sandbox.sandboxsdk import environments

from sandbox.projects.common.solomon import upload_to_solomon


# Resource types
class ToponymAfishaEventsBuilderExecutable(sdk2.Resource):
    """
        Utility that makes afisha events snippets for toponyms and uploads to YT
    """
    executable = True
    releasable = True
    releasers = resource_releasers.geosearch_releasers


# Main task
class ToponymAfishaEventsBuilder(ReleaseToNannyTask2, sdk2.Task):
    '''Task for building index for orgs collections search'''

    class Parameters(sdk2.Parameters):
        snippets_builder_executable = sdk2.parameters.Resource(
            'snippets_builder_executable',
            resource_type=ToponymAfishaEventsBuilderExecutable,
            required=True
        )

        afisha_table = sdk2.parameters.String('Path to afisha schedules', default='//home/afisha/export/production/schedules', required=True)

        out_dir = sdk2.parameters.String('Out dir', required=True)

        geobase_snapshot = sdk2.parameters.Resource(
            'Geobase 5 snapshot',
            resource_type=gbr.GEODATA5BIN_STABLE,
            required=True
        )

        dump_to_indexerproxy_executable = sdk2.parameters.Resource(
            'dump_to_indexerproxy executable',
            resource_type=resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE,
        )

        with sdk2.parameters.String('Environment', required=True, multiline=True) as environment:
            environment.values['stable'] = environment.Value('stable')
            environment.values['testing'] = environment.Value('testing')

        send_stats = sdk2.parameters.Bool('Send stats to Solomon', default=True)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('saaspy'),
        ]

    def execute_cmd(self, executable_parameter, cmd_params):
        executable = str(sdk2.ResourceData(executable_parameter).path)
        logging.info('Run command: {}'.format(' '.join([executable] + cmd_params)))
        with sdk2.helpers.ProcessLog(self, logger='cmd_executor') as l:
            sp.check_call([executable] + cmd_params, stdout=l.stdout, stderr=l.stderr)

    def create_sensors(self, data):
        sensors = []
        timestamp = int(datetime.now().strftime('%s'))
        for k, v in data.items():
            sensors.append({
                'labels': {
                    'sensor': k,
                },
                'ts': timestamp,
                'value': float(v),
            })
        return sensors

    def create_sensors_from_file(self, filename):
        return self.create_sensors(json.load(open(filename)))

    def on_execute(self):
        import yt.wrapper as yt
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        solomon_common_labels = {
            'project': 'toponym_afisha_events_snippets',
            'cluster': 'main',
            'service': self.Parameters.environment,
        }

        logging.info('Building snippets...')
        self.execute_cmd(
            self.Parameters.snippets_builder_executable,
            [
                '--afisha_table', self.Parameters.afisha_table,
                '--snippets_dir', self.Parameters.out_dir,
                '--geobase_file', str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path),
                '--proxy', 'hahn.yt.yandex.net',
                '--out_stats', 'builder_stats.json',
            ]
        )

        if self.Parameters.send_stats:
            upload_to_solomon(solomon_common_labels, self.create_sensors_from_file('builder_stats.json'))

        logging.info('Uploading snippets to saas-kv...')
        if self.Parameters.dump_to_indexerproxy_executable:
            for t in ['events', 'cinema']:
                dumper_log_path = self.Parameters.out_dir + '/saas_snippets_%s_upload.err' % t
                self.execute_cmd(
                    self.Parameters.dump_to_indexerproxy_executable,
                    [
                        '--mode', 'mr',
                        '-k', sdk2.Vault.data('GEOMETA-SEARCH', 'saas_afisha_events_toponyms_snippet_key'),
                        '-h', 'saas-indexerproxy-maps-kv.yandex.net',
                        '--mr-server', 'hahn.yt.yandex.net',
                        '--mr-input', self.Parameters.out_dir + '/saas_snippets_%s' % t,
                        '--mr-processor', 'cards_geo',
                        '--action', 'atModify',
                        '--mr-jobs', '1000',
                        '--mr-output', dumper_log_path,
                        '--prefix', '1',
                        '--mr-processor-env', 'toponym_afisha_%s/1.x' % t,
                    ]
                )
                if self.Parameters.send_stats:
                    yt.config['token'] = yt_token
                    yt.config['proxy']['url'] = 'hahn.yt.yandex.net'
                    upload_to_solomon(solomon_common_labels, self.create_sensors({
                        'dumper_errors_' + t: yt.table_commands.row_count(dumper_log_path),
                    }))
        else:
            logging.info('No dump_to_indexerproxy_executable were provided')

        logging.info('Done')
