import logging
import os

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.projects.resource_types.releasers as resource_releasers
from sandbox.projects import resource_types
from sandbox.projects.common.nanny.nanny import ReleaseToNannyTask2
from sandbox.projects.common.solomon import upload_to_solomon, create_sensors, create_sensors_from_file
from sandbox.projects.geobase.Geodata5BinStable import resource as gbr

from sandbox.sandboxsdk import environments


# Resource types
class ToponymBestCompaniesBuilderExecutable(sdk2.Resource):
    """
        Utility that makes afisha events snippets for toponyms and uploads to YT
    """
    executable = True
    releasable = True
    releasers = resource_releasers.geosearch_releasers


# Main task
class ToponymBestCompaniesBuilder(ReleaseToNannyTask2, sdk2.Task):
    '''Task for building index for orgs collections search'''

    class Parameters(sdk2.Parameters):
        snippets_builder_executable = sdk2.parameters.Resource(
            'snippets_builder_executable',
            resource_type=ToponymBestCompaniesBuilderExecutable,
            required=True
        )

        geometasearch = sdk2.parameters.String('Geometasearch address', default='http://addrs-testing.search.yandex.net/search/stable', required=True)

        add_cgi_params = sdk2.parameters.String('Additional cgi parameters', default='')

        out_table = sdk2.parameters.String('Out snippets table', required=True)

        geobase_snapshot = sdk2.parameters.Resource(
            'Geobase 5 snapshot',
            resource_type=gbr.GEODATA5BIN_STABLE,
            required=True
        )

        dump_to_indexerproxy_executable = sdk2.parameters.Resource(
            'dump_to_indexerproxy executable',
            resource_type=resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE,
        )

        verbose = sdk2.parameters.Bool('Verbose', default=False)

        geoids = sdk2.parameters.String('Build for geoids', default='')

        with sdk2.parameters.String('Environment', required=True, multiline=True) as environment:
            environment.values['stable'] = environment.Value('stable')
            environment.values['testing'] = environment.Value('testing')

        send_stats = sdk2.parameters.Bool('Send stats to Solomon', default=True)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment('saaspy'),
        ]

    def execute_cmd(self, executable_parameter, cmd_params, log_name):
        executable = str(sdk2.ResourceData(executable_parameter).path)
        logging.info('Run command: {}'.format(' '.join([executable] + cmd_params)))
        with sdk2.helpers.ProcessLog(self, logger=log_name) as l:
            sp.check_call([executable] + cmd_params, stdout=l.stdout, stderr=l.stderr)

    def on_execute(self):
        import yt.wrapper as yt
        yt_token = sdk2.Vault.data(self.owner, 'yt-token')

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = yt_token
        os.environ['YT_PROXY'] = 'hahn.yt.yandex.net'
        os.environ['YT_LOG_LEVEL'] = 'INFO'

        solomon_common_labels = {
            'project': 'toponym_best_companies_builder',
            'cluster': 'main',
            'service': self.Parameters.environment,
        }

        logging.info('Building snippets...')
        self.execute_cmd(
            self.Parameters.snippets_builder_executable,
            [
                '--geometasearch', self.Parameters.geometasearch,
                '--add_params', self.Parameters.add_cgi_params,
                '--snippets_table', self.Parameters.out_table,
                '--geobase_file', str(sdk2.ResourceData(self.Parameters.geobase_snapshot).path),
                '--proxy', 'hahn.yt.yandex.net',
                '--out_stats', 'builder_stats.json',
                '--geoids', self.Parameters.geoids,
            ] + (['--verbose'] if self.Parameters.verbose else []),
            'snippets_builder'
        )

        if self.Parameters.send_stats:
            upload_to_solomon(solomon_common_labels, create_sensors_from_file('builder_stats.json'))

        logging.info('Uploading snippets to saas-kv...')
        if self.Parameters.dump_to_indexerproxy_executable:
            dumper_log_path = self.Parameters.out_table + '_upload.err'
            self.execute_cmd(
                self.Parameters.dump_to_indexerproxy_executable,
                [
                    '--mode', 'mr',
                    '-k', sdk2.Vault.data('GEOMETA-SEARCH', 'saas_related_places_snippets_key'),
                    '-h', 'saas-indexerproxy-maps-kv.yandex.net',
                    '--mr-server', 'hahn.yt.yandex.net',
                    '--mr-input', self.Parameters.out_table,
                    '--mr-processor', 'cards_geo',
                    '--action', 'atModify',
                    '--mr-jobs', '1000',
                    '--mr-output', dumper_log_path,
                    '--prefix', '1',
                    '--mr-processor-env', 'related_places/1.x'
                ],
                'dump_to_indexerproxy'
            )
            if self.Parameters.send_stats:
                yt.config['token'] = yt_token
                yt.config['proxy']['url'] = 'hahn.yt.yandex.net'
                upload_to_solomon(solomon_common_labels, create_sensors({
                    'dumper_errors': yt.table_commands.row_count(dumper_log_path),
                }))
        else:
            logging.info('No dump_to_indexerproxy_executable were provided')

        logging.info('Done')
