import os.path as path
import time
import datetime

import sandbox.sdk2 as sdk2

import sandbox.common.types.task as ctt

from sandbox.projects.geosearch.SendToFerryman import CommonFerrymanParameters, SendTablesToFerryman
from sandbox.projects.geosearch.CleanupYtFolder import CommonCleanupParameters, CleanupYtFolder
from sandbox.projects.geosearch.ValidateYtTable import CommonValidateParameters, ValidateYtTable

from sandbox.projects.yql.RunYQL2 import RunYQL2


class UpdateAfishaFts(sdk2.Task):
    """Update afisha data in full text search (SaaS)"""

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.RadioGroup('Yt host') as yt_host:
            yt_host.values['hahn'] = yt_host.Value(value='Hahn', default=True)
            yt_host.values['banach'] = yt_host.Value(value='Banach')

        with sdk2.parameters.Group('Input yt parameters') as input_yt_block:
            afisha_table = sdk2.parameters.String('Yt path to afisha schedules', required=True)

        with sdk2.parameters.Group('Output yt parameters') as output_yt_block:
            history_folder = sdk2.parameters.String('Yt path to folder with results', required=True)

        ferryman_parameters = CommonFerrymanParameters()
        cleaner_parameter = CommonCleanupParameters()
        validator_parameters = CommonValidateParameters()

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def init_context(self):
        self.Context.timestamp = int(time.mktime(datetime.datetime.utcnow().timetuple()))
        self.Context.result_table = path.join(self.Parameters.history_folder, str(self.Context.timestamp))

    def prepare_data(self):
        with open(path.join(path.dirname(path.abspath(__file__)), 'prepare_data.sql')) as query_file:
            query = query_file.read()
        task = RunYQL2(
            self,
            description='Prepare data with afisha schedule for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            query=query,
            custom_placeholders={
                '%CLUSTER%': self.Parameters.yt_host,
                '%OUTPUT_TABLE%': self.Context.result_table,
                '%INPUT_TABLE%': self.Parameters.afisha_table,
            },
            trace_query=True,
        )
        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def validate_data(self):
        task = ValidateYtTable(
            self,
            description='Validate data with afisha schedule for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            validate_rows_diff=self.Parameters.validate_rows_diff,
            validate_rows_diff_percent=self.Parameters.validate_rows_diff_percent,
            validate_mean_row_size_diff=self.Parameters.validate_mean_row_size_diff,
            validate_mean_row_size_diff_percent=self.Parameters.validate_mean_row_size_diff_percent,
            yt_host=self.Parameters.yt_host,
            path=self.Context.result_table,
        )
        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def send_to_ferryman(self):
        task = SendTablesToFerryman(
            self,
            description='Send data with afisha schedule to ferryman for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            ferryman_pen=self.Parameters.ferryman_pen,
            namespace=self.Parameters.namespace,
            is_delta=self.Parameters.is_delta,
            table=self.Context.result_table,
            timestamp=self.Context.timestamp * 10 ** 6,
            wait=True,
        )
        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def clean_folders(self):
        task = CleanupYtFolder(
            self,
            description='Remove old tables for task {}'.format(self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=False,
            yt_host=self.Parameters.yt_host,
            path=self.Parameters.history_folder,
            history_size=self.Parameters.history_size,
        )
        task.enqueue()

        raise sdk2.WaitTask([task.id], ctt.Status.Group.SUCCEED, wait_all=True)

    def on_execute(self):
        with self.memoize_stage.init_context:
            self.init_context()

        with self.memoize_stage.prepare_data:
            self.prepare_data()

        with self.memoize_stage.validate_data:
            self.validate_data()

        with self.memoize_stage.send_to_ferryman:
            self.send_to_ferryman()

        with self.memoize_stage.clean_folder:
            self.clean_folders()
