# -*- coding: utf-8 -*
import os
import logging
import subprocess
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.projects import resource_types as rtypes
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.common.geosearch.startrek import StartrekClient


SNIPPETS_PARAMS = {
    'photo': {'key': '5a5870264eeaebc4a268887e94683139',
              'mr_processor': 'full_yt_row',
              'mr_processor_env': 'snips',
              'rty_adapter': None,
              'prefix': 0},
    'ugc': {'key': 'e98c785bf1c0145f5dbf399a064b51c1',
            'mr_processor': 'sprav_ugc_processor',
            'mr_processor_env': None,
            'rty_adapter': 'json_ref',
            'prefix': None}
    }


class UploadAddrsSnippetPhotos(sdk2.Task):
    '''
        Upload snippet photos to saas-kv
    '''

    class Parameters(sdk2.task.Parameters):
        kill_timeout = 86400
        snippets_type = sdk2.parameters.String('Snippets type',
                                               default_value='photo',
                                               required=False)
        snippets_table = sdk2.parameters.String('Snippets table',
                                                required=False)
        delta_table = sdk2.parameters.String('Snippets delta table',
                                             required=False)
        log_table = sdk2.parameters.String('Log table',
                                           required=True,
                                           default_value='//tmp/altay_photo_snippets.log')
        indexer_proxy = sdk2.parameters.String('Indexer proxy',
                                               required=True,
                                               default_value='saas-indexerproxy-maps-kv.yandex.net')
        startrek_task = sdk2.parameters.String('Startrek task',
                                               required=False)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = (environments.PipEnvironment('yandex-yt'),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def get_binary(self):
        dump_to_indexerproxy_resource = sdk2.Resource[rtypes.DUMP_TO_INDEXERPROXY_EXECUTABLE].find(attrs={'released': 'stable'}).first()
        dump_to_indexerproxy_path = sdk2.ResourceData(dump_to_indexerproxy_resource).path
        return dump_to_indexerproxy_path

    def upload_snippets(self, proc_env, snippets_table_path, action):
        proc_env['MR_RUNTIME'] = 'YT'
        cmd_tpl = ('{binary} '
                   '--mode yt '
                   '-k {service_key} '
                   '-h {indexer_proxy_url} '
                   '--mr-server "hahn.yt.yandex.net:80" '
                   '--mr-input {snippets_table_path} '
                   '--mr-output {log_table_path} '
                   '--mr-processor {mr_processor} '
                   '--action={action} ')
        if SNIPPETS_PARAMS[self.Parameters.snippets_type]['mr_processor_env']:
            cmd_tpl = '{cmd} --mr-processor-env {env} '.format(cmd=cmd_tpl,
                                                               env=SNIPPETS_PARAMS[self.Parameters.snippets_type]['mr_processor_env'])
        if SNIPPETS_PARAMS[self.Parameters.snippets_type]['prefix'] is not None:
            cmd_tpl = '{cmd} --prefix {pref} '.format(cmd=cmd_tpl,
                                                      pref=SNIPPETS_PARAMS[self.Parameters.snippets_type]['prefix'])
        if SNIPPETS_PARAMS[self.Parameters.snippets_type]['rty_adapter']:
            cmd_tpl = '{cmd} --rty_adapter {adapter} '.format(cmd=cmd_tpl,
                                                              adapter=SNIPPETS_PARAMS[self.Parameters.snippets_type]['rty_adapter'])
        cmd = cmd_tpl.format(binary=self.indexer_proxy_dumper_executable,
                             service_key=SNIPPETS_PARAMS[self.Parameters.snippets_type]['key'],
                             indexer_proxy_url=self.Parameters.indexer_proxy,
                             snippets_table_path=snippets_table_path,
                             log_table_path=self.Parameters.log_table,
                             mr_processor=SNIPPETS_PARAMS[self.Parameters.snippets_type]['mr_processor'],
                             action=action)
        log_file_path = get_logs_folder() + '/%s_snippets.out.txt' % action
        with open(log_file_path, 'w') as log_file:
            try:
                logging.info('Running %s' % cmd)
                subprocess.check_call(cmd,
                                      shell=True,
                                      env=proc_env,
                                      stdout=log_file,
                                      stderr=subprocess.STDOUT)
            except subprocess.CalledProcessError as err:
                logging.info('Uploading snippets failed')
                logging.info('Details: %s' % err)
                raise SandboxTaskFailureError('Uploading snippets failed')

    def on_execute(self):
        proc_env = os.environ.copy()
        yt_token = sdk2.Vault.data('GEOMETA-SEARCH', 'yt-token')
        startrek_token = sdk2.Vault.data('robot-geosearch',
                                         'robot_geosearch_startrek_token')
        self.startrek = StartrekClient(startrek_token)
        self.indexer_proxy_dumper_executable = self.get_binary()
        proc_env['YT_TOKEN'] = yt_token
        snippets_path = self.Parameters.snippets_table
        with self.memoize_stage._FULL_STATE_UPLOAD(commit_on_entrance=False):
            if self.Parameters.startrek_task:
                comment = ('((https://yt.yandex-team.ru/hahn/#page=navigation&path={snippets_path} {snippets_type} snippets)) '
                           'upload task:  https://sandbox.yandex-team.ru/task/{task_id}').format(snippets_path=snippets_path,
                                                                                                 snippets_type=self.Parameters.snippets_type,
                                                                                                 task_id=self.id)
                self.startrek.add_comment(self.Parameters.startrek_task,
                                          comment)
            self.upload_snippets(proc_env,
                                 snippets_path,
                                 action='atModify')
        with self.memoize_stage._DELETE_DELTA(commit_on_entrance=False):
            if self.Parameters.delta_table:
                self.startrek.add_comment(self.Parameters.startrek_task,
                                          comment)
                self.upload_snippets(proc_env,
                                     self.Parameters.delta_table,
                                     action='atDelete')
                delta_comment = 'Deleting delta finished'
                self.startrek.add_comment(self.Parameters.startrek_task,
                                          delta_comment)
