# -*- coding: utf-8 -*-

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.sandboxsdk import environments
from sandbox.projects.geosearch import resource_types as geo_types
from sandbox.projects import resource_types

import os
import time


class UploadInnerOrgs(sdk2.Task):
    """
    Publish inner orgs snippets to SaaS
    """

    class Parameters(sdk2.Task.Parameters):
        server = sdk2.parameters.String("YT server", default='hahn', required=True)
        preparer = sdk2.parameters.Resource(
            'Dump preparer',
            resource_type=geo_types.INNER_ORGS_PREPARER,
            required=True
        )
        skip_prepare = sdk2.parameters.Bool("Skip prepare phase", default=False)
        dumper = sdk2.parameters.Resource(
            'SaaS uploader',
            resource_type=resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE,
            required=True
        )
        indexer = sdk2.parameters.String("SaaS indexer url", default='saas-indexerproxy-maps-kv.yandex.net', required=True)
        src_dir = sdk2.parameters.String("Yt dir with snippet data tables", default='//home/geosearch/inner_orgs/', required=True)
        dumps_dir = sdk2.parameters.String("Yt dir with SaaS dumps tables", default='//home/geosearch/mesherin/', required=True)
        companies_top_size = sdk2.parameters.Integer("Companies top size", default=0)

    class Requirements(sdk2.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt', '0.7.34.post0', use_wheel=True),
        ]

    def on_execute(self):
        preparer = str(sdk2.ResourceData(self.Parameters.preparer).path)
        token = sdk2.Vault.data('mesherin', 'YQL_TOKEN')
        oids_dump = self.Parameters.dumps_dir + 'inner_orgs_saas_dump'
        oids_dump_light = self.Parameters.dumps_dir + 'inner_orgs_light_saas_dump'
        categ_dump = self.Parameters.dumps_dir + 'inner_categs_saas_dump'
        prep_err = os.path.join(str(self.log_path()), 'upload_inner_orgs.err')
        exec_args = [preparer, '-p', self.Parameters.server, '-s', self.Parameters.src_dir, '-d', self.Parameters.dumps_dir]

        if self.Parameters.companies_top_size:
            exec_args.extend(['--companies-top', str(self.Parameters.companies_top_size)])

        os.environ['MR_RUNTIME'] = 'YT'
        os.environ['YT_TOKEN'] = token

        if not self.Parameters.skip_prepare:
            sp.check_call(exec_args, stderr=open(prep_err, 'w'))

        indexer = self.Parameters.indexer
        dumper = str(sdk2.ResourceData(self.Parameters.dumper).path)

        oid_dump_err = os.path.join(str(self.log_path()), 'dump_to_indexerproxy_oids.err')
        oid_upload_params = [
            dumper,
            '--mode', 'mr',
            '-k', '0965dcc45da238136de79a1b71a89133',
            '-h', indexer,
            '--mr-server', self.Parameters.server,
            '--mr-input', oids_dump,
            '--mr-output', oids_dump + '.err',
            '--mr-processor', 'cards_geo',
            '--mr-processor-env', 'companies_inside/1.x',
            '--prefix', '1',
            '--action', 'atModify',
            '-l', '4',
            '--deadline', str(int(time.time() + 259200) / 60),  # ES snippet ttl is 72 hours
        ]
        sp.check_call(oid_upload_params, stderr=open(oid_dump_err, 'w'))

        oid_dump_err = os.path.join(str(self.log_path()), 'dump_to_indexerproxy_oids_light.err')
        oid_upload_params = [
            dumper,
            '--mode', 'mr',
            '-k', '0965dcc45da238136de79a1b71a89133',
            '-h', indexer,
            '--mr-server', self.Parameters.server,
            '--mr-input', oids_dump_light,
            '--mr-output', oids_dump_light + '.err',
            '--mr-processor', 'cards_geo',
            '--mr-processor-env', 'companies_inside_light/1.x',
            '--prefix', '1',
            '--action', 'atModify',
            '-l', '4',
            '--deadline', str(int(time.time() + 259200) / 60),  # ES snippet ttl is 72 hours
        ]
        sp.check_call(oid_upload_params, stderr=open(oid_dump_err, 'w'))

        categ_dump_err = os.path.join(str(self.log_path()), 'dump_to_indexerproxy_categs.err')
        categ_upload_params = [
            dumper,
            '--mode', 'mr',
            '-k', '0965dcc45da238136de79a1b71a89133',
            '-h', indexer,
            '--mr-server', self.Parameters.server,
            '--mr-input', categ_dump,
            '--mr-output', categ_dump + '.err',
            '--mr-processor', 'cards_geo',
            '--mr-processor-env', 'categories_inside/1.x',
            '--prefix', '1',
            '--action', 'atModify',
            '-l', '4',
            '--deadline', str(int(time.time() + 259200) / 60),  # ES snippet ttl is 72 hours
        ]
        sp.check_call(categ_upload_params, stderr=open(categ_dump_err, 'w'))
