from os import path

import sandbox.sdk2 as sdk2
import sandbox.sandboxsdk.environments as environments

import sandbox.common.errors as errors

PREVIOUS_VALID_ROW_COUNT_ATTRIBUTE = 'previous_valid_row_count'
PREVIOUS_MEAN_ROW_SIZE_ATTRIBUTE = 'previous_mean_row_size'


def is_diff_valid(current_value, previous_value, percent):
    return abs(float(current_value - previous_value)) / previous_value < percent


class CommonValidateParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Yt validator parameters') as validator_block:
        validate_rows_diff = sdk2.parameters.Bool('Validate diff size')
        with validate_rows_diff.value[True]:
            validate_rows_diff_percent = sdk2.parameters.Float('Relative change in row count',
                                                               required=True, default=0.05)

        validate_mean_row_size_diff = sdk2.parameters.Bool('Validate mean row size')
        with validate_mean_row_size_diff.value[True]:
            validate_mean_row_size_diff_percent = sdk2.parameters.Float('Relative change in mean row size',
                                                                        required=True, default=0.01)


class ValidateYtTable(sdk2.Task):
    """Validate table by directory attributes"""

    class Parameters(CommonValidateParameters):
        with sdk2.parameters.Group('Yt parameters') as yt_block:
            with sdk2.parameters.RadioGroup('Yt host') as yt_host:
                yt_host.values['hahn'] = yt_host.Value(value='Hahn', default=True)
                yt_host.values['banach'] = yt_host.Value(value='Banach')

            path = sdk2.parameters.String('Table path', required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        import yt.wrapper as yt
        yt.config['token'] = sdk2.Vault.data(self.owner, 'yt-token')
        yt.config['proxy']['url'] = '{}.yt.yandex.net'.format(self.Parameters.yt_host)

        current_row_count = yt.row_count(self.Parameters.path)
        mean_row_size = float(yt.get_attribute(self.Parameters.path, 'uncompressed_data_size')) / current_row_count
        directory = path.dirname(self.Parameters.path)

        with self.memoize_stage.validate:
            #  validate row count
            if self.Parameters.validate_rows_diff:
                previous_valid_row_count = yt.get_attribute(directory, PREVIOUS_VALID_ROW_COUNT_ATTRIBUTE, None)

                if previous_valid_row_count is None:
                    is_row_diff_valid = True
                else:
                    is_row_diff_valid = is_diff_valid(current_row_count, previous_valid_row_count,
                                                      self.Parameters.validate_rows_diff_percent)

            else:
                is_row_diff_valid = True

            # validate mean row size
            if self.Parameters.validate_mean_row_size_diff:
                previous_mean_row_size = yt.get_attribute(directory, PREVIOUS_MEAN_ROW_SIZE_ATTRIBUTE, None)

                if previous_mean_row_size is None:
                    is_mean_row_size_diff_valid = True
                else:
                    is_mean_row_size_diff_valid = is_diff_valid(mean_row_size, previous_mean_row_size,
                                                                self.Parameters.validate_mean_row_size_diff_percent)

            else:
                is_mean_row_size_diff_valid = True

            # raise error is not valid
            error_messages = list()
            if not is_row_diff_valid:
                error_messages.append('Row count diff is not valid')

            if not is_mean_row_size_diff_valid:
                error_messages.append('Mean row size diff is not valid')

            if error_messages:
                raise errors.TaskError('\n'.join(error_messages))

        yt.set_attribute(directory, PREVIOUS_VALID_ROW_COUNT_ATTRIBUTE, current_row_count)
        yt.set_attribute(directory, PREVIOUS_MEAN_ROW_SIZE_ATTRIBUTE, mean_row_size)
