import os
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

import sandbox.common.types.resource as ctr
from sandbox.common import errors


class YtTransferManagerExecutable(sdk2.Resource):
    executable = True
    releasable = True
    releasers = ['zhshishkin']


class CommonYtTransferParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Transfer manager parameters') as yt_transfer_manager_block:
        yt_transfer_manager_resource_id = sdk2.parameters.LastReleasedResource(
            'Transfer manager',
            resource_type=YtTransferManagerExecutable,
            state=(ctr.State.READY,),
            required=True,
        )


class YtTransfer(sdk2.Task):
    """Transfer tables between yt clusters"""

    class Parameters(CommonYtTransferParameters):
        with sdk2.parameters.Group('YT src parameters') as src_parameters:
            with sdk2.parameters.RadioGroup('Src cluster') as src_cluster:
                src_cluster.values['hahn'] = src_cluster.Value(value='Hahn', default=True)
                src_cluster.values['banach'] = src_cluster.Value(value='Banach')
                src_cluster.values['arnold'] = src_cluster.Value(value='Arnold')
                src_cluster.values['bohr'] = src_cluster.Value(value='Bohr')
                src_cluster.values['landau'] = src_cluster.Value(value='Landau')
                src_cluster.values['markov'] = src_cluster.Value(value='Markov')

            src_table = sdk2.parameters.String(
                'Src table',
                required=True
            )

        with sdk2.parameters.Group('YT dst parameters') as dst_parameters:
            with sdk2.parameters.RadioGroup('Dst cluster') as dst_cluster:
                dst_cluster.values['hahn'] = dst_cluster.Value(value='Hahn', default=True)
                dst_cluster.values['banach'] = dst_cluster.Value(value='Banach')
                dst_cluster.values['arnold'] = dst_cluster.Value(value='Arnold')
                dst_cluster.values['bohr'] = dst_cluster.Value(value='Bohr')
                dst_cluster.values['landau'] = dst_cluster.Value(value='Landau')
                dst_cluster.values['markov'] = dst_cluster.Value(value='Markov')

            dst_table = sdk2.parameters.String(
                'Dst table',
                required=True
            )

    class Requirements(sdk2.Task.Requirements):
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        with self.memoize_stage.transfer_phase(commit_on_entrance=False):
            transfer_manager = self.Parameters.yt_transfer_manager_resource_id

            if transfer_manager is None:
                raise errors.TaskError('No executable founded')
            transfer_manager = sdk2.ResourceData(transfer_manager)

            logging.info('Start run')
            with sdk2.helpers.ProcessLog(self, logger='run transfer_manager') as pl:
                env = os.environ.copy()
                env['YT_TOKEN'] = sdk2.Vault.data(self.owner, 'yt-token')
                run = [
                    str(transfer_manager.path), 'add-task', '--sync',
                    '--src-cluster', self.Parameters.src_cluster,
                    '--dst-cluster', self.Parameters.dst_cluster,
                    '--src-table', self.Parameters.src_table,
                    '--dst-table', self.Parameters.dst_table,
                ]
                ret = sp.Popen(run, stdout=pl.stdout, stderr=sp.STDOUT, env=env).wait()
                if ret:
                    raise errors.TaskError('run is failed')
