# -*- coding: utf-8 -*
import os
import json
import jinja2
import logging
import requests

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia


MINUTE = 60
HOUR = 60 * MINUTE
TIMEDELTA_ALERT_SETTINGS = {
    'task_manager.json': {
        'WARN': 24 * HOUR,
        'CRIT': 36 * HOUR
    },
    'fast_task_manager.json': {
        'WARN': 3 * HOUR,
        'CRIT': 6 * HOUR
    },
    'realty_task_manager.json': {
        'WARN': 2 * HOUR,
        'CRIT': 4 * HOUR
    },
    'base_sync.json': {
        'WARN': 24 * HOUR,
        'CRIT': 60 * HOUR
    },
    'objects.json': {
        'WARN': 6 * HOUR,
        'CRIT': 12 * HOUR
    }

}

TEMPLATE = """
<% set tags = {'itype': 'geosnippets', 'ctype': 'test', 'prj': 'addrs'} %>
<% set base = {'mgroups': ['ASEARCH'], 'name_tmpl': '{signal_screened}'} %>
    [
    {%- for conf_name in data.configs -%}
        {%- set timedelta_threshold = data.timedelta_thresholds[conf_name] -%}
        {%- set params = data.configs[conf_name] -%}
        {% for snippet in params %}
            <% with juggler_base = { 'aggregator': 'logic_or',
                'aggregator_kwargs': {'nodata_mode': 'force_ok',
                                      'unreach_mode': 'force_ok',
                                      'unreach_service': [{'check': 'yasm_alert:virtual-meta'}]},
                'children': [{'host': 'yasm_alert',
                              'service': None,
                             }],
                'flaps': {'boost': 120, 'critical': 60, 'stable': 30},
                'host': 'addrs',
                'methods': [],
                'namespace': 'addrs',
                'refresh_time': 5} %>
                    << Alert(base=base,
                             signal='push-{{ snippet.snippet_name }}_timedelta_axxx',
                             tags=tags,
                             warn=[{{ timedelta_threshold.WARN }}, {{ timedelta_threshold.CRIT }}],
                             crit=[{{ timedelta_threshold.CRIT }}, None],
                             juggler_base=juggler_base,
                             juggler_service_tmpl='{{ snippet.snippet_name }}_age_seconds',
                             flaps={'critical': 240, 'stable': 30, 'boost': 60},
                             juggler_tags=['a_itype_geosnippets', 'a_ctype_test', 'a_prj_addrs'],
                             shorten_name=True) >>,
            <% endwith %>
            <% with juggler_base = { 'aggregator': 'logic_or',
                'aggregator_kwargs': {'nodata_mode': 'force_ok',
                                      'unreach_mode': 'force_ok',
                                      'unreach_service': [{'check': 'yasm_alert:virtual-meta'}]},
                'children': [{'host': 'yasm_alert',
                              'service': None,
                             }],
                'flaps': {'boost': 120, 'critical': 60, 'stable': 30},
                'host': 'addrs',
                'methods': [],
                'namespace': 'addrs',
                'notifications': [{'template_kwargs': {'login': {{ snippet.notifications }},
                                                       'method': ['email', 'telegram'],
                                                       'status': ['CRIT']},
                                   'template_name': 'on_status_change'}],
                'refresh_time': 5} %>
                    << TrendAlert(base=base,
                                  signal='push-{{ snippet.snippet_name }}_errors_axxx',
                                  trend="up",
                                  interval=1800,
                                  crit_perc=40,
                                  warn_perc=20,
                                  tags=tags,
                                  juggler_base=juggler_base,
                                  juggler_service_tmpl='{{ snippet.snippet_name }}_errors_inc_perc',
                                  flaps={'critical': 240, 'stable': 30, 'boost': 60},
                                  juggler_tags=['a_itype_geosnippets', 'a_ctype_test', 'a_prj_addrs'],
                                  shorten_name=True) >>,
            <% endwith %>
        {% endfor %}
    {%- endfor -%}
    ]
"""


class AddrsSnippetsAlertsGenerator(sdk2.Task):
    '''
        Generates Yasm alert template
    '''

    class Parameters(sdk2.task.Parameters):
        configs = sdk2.parameters.List('Geosearch snippets config files',
                                       required=True)
        yasm_key = sdk2.parameters.String('Yasm key',
                                          default='geosnippets',
                                          required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 2048

        class Caches(sdk2.Requirements.Caches):
            pass

    def _simplify_name(self, name):
        return name.replace('/', '_').replace('.', '_')

    def _read_config(self, arcadia_path):
        values = []
        checkout_path = Arcadia.export(arcadia_path, os.path.basename(arcadia_path))
        config = json.load(open(checkout_path))
        for value in config.itervalues():
            if not value.get('test'):
                if 'timestamp_field' not in value:
                    try:
                        notify = ['@svc_geosearch2_dutywork'] + value.get('yasm_alerts', [])
                        values.append({'snippet_name': self._simplify_name(value.get('snippet_name')),
                                       'notifications': [str(person) for person in notify]})
                    except Exception:
                        logging.exception('Failed on {snippet}'.format(snippet=value))
        return values

    def _get_configs(self):
        configs = {}
        for arcadia_path in self.Parameters.configs:
            config = self._read_config(arcadia_path)
            configs.update({os.path.basename(arcadia_path): config})
        data = {'timedelta_thresholds': TIMEDELTA_ALERT_SETTINGS,
                'configs': configs}
        logging.info('Data for template rendering: %s' % data)
        return data

    def render_template(self):
        data = self._get_configs()
        template = jinja2.Template(TEMPLATE)
        alerts = template.render({'data': data})
        logging.info('Alerts template:\n %s' % alerts)
        return alerts

    def delete_template(self):
        url = 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/delete?key={key}'.format(key=self.Parameters.yasm_key)
        logging.info('Trying to delete panel %s' % self.Parameters.yasm_key)
        try:
            response = requests.post(url)
            logging.info(response.text)
        except Exception:
            pass

    def create_template(self):
        url = 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/create'
        data = {
            'key': self.Parameters.yasm_key,
            'owners': ['karas-pv'],
            'content': self.Context.template,
            'is_periodic': True}
        logging.info('Trying to create panel %s' % data.get('key'))
        try:
            response = requests.post(url, data=json.dumps(data))
            logging.info(response.text)
        except Exception:
            pass

    def apply_template(self):
        url = 'https://yasm.yandex-team.ru/srvambry/tmpl/alerts/apply/{key}'.format(key=self.Parameters.yasm_key)
        logging.info('Applying alerts from panel %s' % self.Parameters.yasm_key)
        try:
            response = requests.post(url)
            logging.info(response.text)
        except Exception:
            pass

    def get_service_name(self, config_name):
        service_names = {
            'task_manager.json': 'daily',
            'fast_task_manager.json': 'fast',
            'realty_task_manager.json': 'realty',
            'base_sync.json': 'base_sync',
            'objects.json': 'objects'
        }
        return 'geosnippets_{}_timedelta'. format(service_names.get(config_name))

    def make_meta_aggregator(self):
        data = {
            'project': 'addrs',
            'host': 'addrs',
            'refresh_time': 5,
            'ttl': 900,
            'aggregator': 'logic_or',
            'aggregator_kwargs': {
                'nodata_mode': 'force_ok',
                'unreach_mode': 'force_ok',
                'unreach_service': {
                    'check': 'yasm_alert:virtual-meta'
                }
            },
            'flaps': {
                'stable_time': 30,
                'critical_time': 240,
                'boost_time': 60
            },
            'notifications': [
                {
                    'template_kwargs': {
                        'login': ['@svc_geosearch2_dutywork'],
                        'method': ['email', 'telegram'],
                        'status': ['CRIT'],
                        'time_start': '11:00',
                        'time_end': '00:00'
                    },
                    'template_name': 'on_status_change'
                }
            ]
        }
        configs = self._get_configs()['configs']
        token = sdk2.Vault.data('robot-geosearch', 'juggler-token')
        for file_name, config in configs.iteritems():
            data.update({'service': self.get_service_name(file_name)})
            children = []
            for snippet in config:
                children.append(
                    {
                        'host': 'yasm_alert',
                        'service': 'geosnippets.push-{}_timedelta_axxx'.format(snippet.get('snippet_name')),
                        'type': 'HOST'
                    }
                )
            data.update({'children': children})
            url = 'https://juggler-api.yandex-team.ru/api/checks/add_or_update?do=1'
            try:
                headers = {'Authorization': 'OAuth {}'.format(token)}
                response = requests.post(url, data=json.dumps(data), headers=headers)
                logging.info('Meta aggregator {} for {} created. API response: {}'.format(
                    self.get_service_name(file_name),
                    file_name,
                    response.content
                ))
            except Exception as err:
                logging.info('Failed to create meta aggregator: {}'.format(err))

    def on_execute(self):
        self.Context.template = self.render_template()
        self.delete_template()
        self.create_template()
        self.apply_template()
        self.make_meta_aggregator()
