# -*- coding: utf-8 -*-

from sandbox import sdk2

import logging
import datetime

import sandbox.common.types.notification as ctn
from sandbox.projects.geosearch.tools import abc_duty

GEOSEARCH_SERVICE_ID = '2016'


def get_duty_day(day):
    today = datetime.date.today()
    if day == 'monday':
        return str(today - datetime.timedelta(days=today.weekday()))
    if day == 'sunday':
        return str(today + datetime.timedelta(days=(7 - today.weekday() - 1)))      # '- 1' to skip current day


def who_is_on_duty():
    abc_token = sdk2.Vault.data('GEOSEARCH_PROD', 'geosearch_abc_token')

    parameters = {
        'date_from': get_duty_day('monday'),
        'date_to': get_duty_day('sunday'),
    }

    try:

        return abc_duty.get_service_duty_logins(
            abc_service_id=GEOSEARCH_SERVICE_ID,
            parameters=parameters,
            abc_token=abc_token,
        )

    except Exception:
        logging.exception()
        return []


def notify_by_telegram(task, event='', message=None):
    try:
        on_duty = who_is_on_duty()
        if not message:
            message = ('Geosearch task {type}#{id} {event}\n'
                       'Details can be found in '
                       'https://sandbox.yandex-team.ru/task/{id}/view').format(type=task.type, id=task.id, event=event)
        logging.info('Telegram notification on {event} will be sent to {on_duty}'.format(event=event, on_duty=on_duty))
        task.server.notification(body=message,
                                 recipients=on_duty,
                                 transport=ctn.Transport.TELEGRAM)
    except Exception:
        logging.exception('Failed to send telegram notifications')


class NotifyGeosearchDuty(object):
    """
        Notifies geosearch duty about task failure (mixin class for sdk2.Task)
    """
    class Parameters(sdk2.task.Parameters):
        notify_duty = sdk2.parameters.Bool('Notify geosearch duty about task failure', default_value=False, do_not_copy=True)

    def on_failure(self, prev_status):
        super(NotifyGeosearchDuty, self).on_failure(prev_status)
        if self.Parameters.notify_duty:
            notify_by_telegram(self, 'failed')

    def on_break(self, prev_status, status):
        super(NotifyGeosearchDuty, self).on_break(prev_status, status)
        if self.Parameters.notify_duty:
            notify_by_telegram(self, 'failed with exception')

    def on_timeout(self, prev_status):
        super(NotifyGeosearchDuty, self).on_timeout(prev_status)
        if self.Parameters.notify_duty:
            notify_by_telegram(self, 'timed out')
