import re
from collections import namedtuple

BetaOptions = namedtuple('BetaOptions', ['source', 'tier', 'path'])

TEMPLATES = {
    'wizard': BetaOptions('wizard', None, '/wizard'),
    'base': BetaOptions('business', 'addrs-base-{:03d}', ''),
    'base_18': BetaOptions('business', 'addrs-base-{:03d}', ''),
    'sprav': BetaOptions('business', 'addrs-base-{:03d}', ''),
    'collections': BetaOptions('collections', None, ''),
    # Nobody overrides metasearch and geosuggest through `source` or `metahost2` params,
    # though these entries are left for compatibility
    'meta': BetaOptions(None, None, ''),
    'geometasearch_upper': BetaOptions(None, None, ''),
    'geometasearch_middle': BetaOptions('middle', None, '/yandsearch'),
    'geosuggest': BetaOptions('geosuggest', None, ''),
}


class SameShardException(Exception):
    pass


def _get_yp_shard_number(text):
    m = re.search(r'\[\/labels\/shard_id\]\="(?P<id>\d+)"', text)
    if m is None:
        raise ValueError('No shard number found in "{}"'.format(text))
    return int(m.group('id'))


def _make_metahost2(source, script, tier=None):
    assert script  # script is either host-port or sd://-expression
    res = ''
    if source:
        res += source
        if tier:
            res += '!' + tier
        res += ':'
    res += script
    return res


def render_experimental_sources_from_yp(beta_type, endpoint_sets):
    template = TEMPLATES.get(beta_type)
    if template is None:
        return []

    search_experimental_sources = []
    processed_shardnumbers = set()

    for cluster, endpoint_sets in endpoint_sets.items():
        for endpoint_set in endpoint_sets:
            tier = None
            if template.tier is not None:
                # sharded source
                shard_number = _get_yp_shard_number(endpoint_set['spec']['podFilter']) - 1  # convert 1-based to 0-based

                if shard_number in processed_shardnumbers:
                    msg = 'Shard {} already processed. Check YP API data'.format(shard_number)
                    raise SameShardException(msg)
                processed_shardnumbers.add(shard_number)

                tier = template.tier.format(shard_number)

            script = 'sd://{cluster}@{endpoint_set_id}{path}'.format(
                cluster=cluster.lower(),
                endpoint_set_id=endpoint_set['meta']['id'],
                path=template.path,
            )
            search_experimental_sources.append(_make_metahost2(template.source, script, tier))

    return search_experimental_sources


def render_experimental_sources_from_nanny(beta_type, instances):
    template = TEMPLATES.get(beta_type)
    if template is None:
        return []
    if template.tier is not None:
        raise ValueError('Sharded services are not supported because the instance list from Nanny does not contain shard numbers')

    search_experimental_sources = []

    for instance in instances:
        script = '{host}:{port}{path}'.format(
            host=instance.get('container_hostname'),
            port=instance.get('port') or 80,
            path=template.path
        )
        search_experimental_sources.append(_make_metahost2(template.source, script))

    return search_experimental_sources
