# -*- coding: utf-8 -*-

import os
import time
import logging
import requests
import datetime

from sandbox import sdk2


BASE_PATH = '//home/geosearch-prod/addrs_base'
TABLES = {'building': os.path.join(BASE_PATH, 'building'),
          'accepting': os.path.join(BASE_PATH, 'accepting'),
          'perf': os.path.join(BASE_PATH, 'perf'),
          'production': os.path.join(BASE_PATH, 'production')}


def _read(stage, client):
    table_path = TABLES.get(stage)
    if not client.exists(table_path):
        return []
    data = client.read_table(table_path, format='json')
    return [d for d in data]


def _write(data, stage, client):
    table_path = TABLES.get(stage)
    if not client.exists(table_path):
        client.create_table(table_path, recursive=True)
    client.write_table(table_path, data, format='json')


def _add(data, stage, client):
    table_path = TABLES.get(stage)
    if not client.exists(table_path):
        client.create_table(table_path, recursive=True)
    table = _read(stage, client)
    for entry in table:
        if 'yt_path' in data and entry['yt_path'] == data['yt_path']:
            return None   # Already exist
    table.append(data)
    logging.info('Trying to write %s to  %s' % (table, TABLES.get(stage)))
    _write(table, stage, client)


def _remove(data, stage, client, remove_old=False):
    content = _read(stage, client)
    for entry in content:
        if entry['yt_path'] == data['yt_path']:
            content.remove(entry)
        if remove_old and int(entry['build_task']) < int(data['build_task']):
            content.remove(entry)
    _write(content, stage, client)


def building(data, client):
    _add(data, 'building', client)


def accepting(data, client):
    _remove(data, 'building', client)
    _add(data, 'accepting', client)


def production(data, client):
    _write([], 'production', client)
    _add(data, 'production', client)
    _remove(data, 'accepting', client, remove_old=True)
    _remove(data, 'building', client, remove_old=True)


def read_stat_table(token, stat_path, field_date, stage):
    date_max = datetime.datetime.now()
    date_min = date_max - datetime.timedelta(days=2)
    date_format = '%Y-%m-%d'
    url = ('https://upload.stat.yandex-team.ru/_api/statreport/json/{path}?'
           'scale=s&'
           'date_min={d_min}&'
           'date_max={d_max}&'
           '_bldr_fill_missing_dates=0').format(path=stat_path,
                                                d_min=date_min.strftime(date_format),
                                                d_max=date_max.strftime(date_format))
    headers = {'Authorization': 'OAuth {token}'.format(token=token)}
    try:
        response = requests.get(url, headers=headers).json()
        return response.get('values')
    except Exception:
        logging.exception('Failed to read data from {stat_path}'.format(stat_path=stat_path))
        return []


def _check_stats(values, field_date, stage):
    if values is not None:
        for value in values:
            if field_date in value and stage in value:
                return True
    return False


def written(token, stat_path, field_date, stage):
    values = read_stat_table(token, stat_path, field_date, stage)
    return _check_stats(values, field_date, stage)


def push_to_stat_table(stat_path, table_path, stage, timestamp=0):
    url = 'https://upload.stat.yandex-team.ru/_api/report/data/{path}?scale=s'.format(path=stat_path)
    date_str = table_path.split('-', 1)[1].split('+')[0]
    if not timestamp:
        timestamp = int(time.time())
    data = {'fielddate': str(date_str),
            'time': str(timestamp),
            'stage': stage}
    token = sdk2.Vault.data('GEOMETA-SEARCH', 'GEOSEARCH_STAT_TOKEN')
    headers = {'Authorization': 'OAuth {token}'.format(token=token)}
    try:
        if not written(token, stat_path, str(date_str), stage):
            response = requests.post(url, json={'data': [data]}, headers=headers)
            logging.info('Posted data on {stage}.\n{resp}'.format(stage=stage, resp=response.json()))
    except Exception:
        logging.exception('Failed to push {stage} data to {table}'.format(stage=stage,
                                                                          table=stat_path))
