# -*- coding: utf-8 -*-

import time
import nanny
import yp_lite
import json
import logging
import requests
from sandbox import sdk2
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.geosearch.tools.misc import retry
from sandbox.projects.geosearch.tools.source_param import (
    render_experimental_sources_from_yp,
    render_experimental_sources_from_nanny,
)

YAPPY_STATUS_URL = 'https://yappy.z.yandex-team.ru/api/yappy.services.Model/getBetaStatus'
YAPPY_DATA_URL = 'https://yappy.z.yandex-team.ru/api/yappy.services.Model/retrieveBeta'
YAPPY_QUOTA_URL = 'http://yappy.z.yandex-team.ru/api/yappy.services.Model/listQuotas'

YAPPY_LIST_BETAS_URL = 'https://yappy.z.yandex-team.ru/api/yappy.services.Model/listBetas'
YAPPY_CREATE_FROM_TEMPLATE_URL = 'https://yappy.z.yandex-team.ru/api/yappy.services.Lineage2/createBetaFromBetaTemplate'
YAPPY_FREEZE_BETA = 'https://yappy.z.yandex-team.ru/api/yappy.services.Model/freezeComponents'
YAPPY_STOP_BETA = 'https://yappy.z.yandex-team.ru/api/yappy.services.Lineage2/deallocateBeta'
YAPPY_DELETE_BETA = 'https://yappy.z.yandex-team.ru/api/yappy.services.Lineage2/deleteBeta'


# {beta_type: {component_id/qota_name: parent_service}}
YAPPY_DATA = {
    'meta': {
        'addrs-upper': 'addrs_upper'
    },
    'geometasearch': {
        'geometasearch_upper': 'addrs_upper_stable',
        'geometasearch_middle': 'addrs_middle_stable'
    },
    'wizard': {
        'addrs-wizard': 'addrs_wizard_yp'
    },
    'base': {
        'addrs-base': 'addrs_base_stable'
    },
    'sprav': {
        'sprav-base': 'addrs_base_stable'
    },
    'collections': {
        'addrs-collections': 'addrs_collections_yp'
    },
    'geosuggest': {
        'geosuggest': 'suggest_maps'
    }
}


def create_session(token):
    session = requests.Session()
    session.headers['Authorization'] = 'OAuth {}'.format(token)
    session.headers['Content-Type'] = 'application/json'
    return session


@retry(tries=5, delay=3)
def get_quota(beta_type, token):
    session = create_session(token)
    data = session.post(YAPPY_QUOTA_URL).json()
    quotas = data.get('objects', [])
    parent_quota_names = []
    for key in YAPPY_DATA.get(beta_type):
        parent_quota_names.append('{b_type}@{b_type}'.format(b_type=key.replace('-', '_')))
    parent_quotas = []
    for quota in quotas:
        if quota.get('name', '') in parent_quota_names:
            parent_quotas.append(quota)
    return parent_quotas


def get_slots(beta_type, token):
    quotas = get_quota(beta_type, token)
    slots = []
    for quota in quotas:
        slots.extend(quota.get('slots', []))
    return slots


@retry(tries=5, delay=3)
def free_slots(beta_type, token):
    session = create_session(token)
    slots = get_slots(beta_type, token)
    logging.info('Slots: %s' % slots)
    response = session.post(YAPPY_LIST_BETAS_URL)
    data = response.json()
    if not data:
        logging.debug('Yappy listBetas response code: %s' % response.status_code)
        logging.debug(response.text)
        raise SandboxTaskFailureError('Could not get beta list from Yappy API')
    beta_name_templates = ['%s-' % component_id for component_id in YAPPY_DATA.get(beta_type).keys()]
    betas = []
    for name_template in beta_name_templates:
        betas.extend([beta for beta in data.get('objects') if name_template in beta.get('name') and beta.get('status') != 'STOPPED'])
    logging.info('Betas: %s' % betas)
    if not betas:
        return True
    if len(betas) < len(slots):
        return True
    return False


def stop_beta(beta_name, token):
    session = create_session(token)
    resp = session.post(YAPPY_STOP_BETA, data=json.dumps({'name': beta_name}))
    logging.info('Stoping {beta} result: {result}'.format(beta=beta_name, result=resp.text))


class YappyBeta(object):

    def __init__(self, branch, tag, beta_type, resources, task_id, nanny_token, yappy_token):
        self.token = yappy_token
        self.session = create_session(self.token)
        self.nanny_cli = nanny.Nanny(nanny_token)
        self.yp_lite = yp_lite.YpLightAPI(nanny_token)
        self.branch = branch.replace('_', '-')
        self.tag = tag.replace('_', '-')
        self.beta_type = beta_type
        self.data = YAPPY_DATA.get(self.beta_type)
        logging.info('Beta type = %s' % self.beta_type)
        self.sandbox_task_id = task_id
        self.name = self.make_beta_name()
        self.resources = resources

    def is_multicomponent(self):
        return len(self.data) > 1

    def make_beta_name(self):
        template = '{}-{}-{}-{}'
        if self.is_multicomponent():
            return template.format(self.beta_type, self.branch, self.tag, self.sandbox_task_id)
        return template.format(self.data.keys()[0], self.branch, self.tag, self.sandbox_task_id)

    def _make_patch_item(self, data, yappy_resources):
        logging.info('Processing %s' % data)
        for resource in yappy_resources:
            yappy_resource = sdk2.Resource[resource]
            if yappy_resource.type == data['resource_type']:
                data['resource_id'] = str(yappy_resource.id)
        yappy_data = {}
        if data.get('resource_id'):
            yappy_data.update({'sandboxResourceId': data.get('resource_id')})
        else:
            task_obj = sdk2.Task[data['task_id']]
            resource_class = sdk2.Resource[data['resource_type']]
            resource = resource_class.find(task=task_obj).first()
            yappy_data.update({'sandboxResourceId': str(resource.id)})
        yappy_data.update({'manageType': 'SANDBOX_RESOURCE'})
        if data['resource_type'] in ['ADDRS_BUSINESS_SHARDMAP', 'ADDRS_BASE_YP_SHARDMAP']:
            yappy_data.update({'manageType': 'SANDBOX_SHARDMAP'})
        if data.get('local_path'):
            yappy_data.update({'localPath': data.get('local_path')})
        return yappy_data

    def _make_yappy_data(self, parent_service, resources):
        nanny_data = self.nanny_cli.get_sandbox_files(parent_service)
        reference_runtime_data = nanny_data.get('sandbox_files')
        try:
            reference_runtime_data.append(nanny_data['sandbox_bsc_shard']['sandbox_shardmap'])
        except Exception:
            pass
        logging.info('reference_runtime_data = %s' % reference_runtime_data)
        patch_data = []
        for reference_resource in reference_runtime_data:
            patch_data.append(self._make_patch_item(reference_resource, resources))
        return {
            'parentExternalId': parent_service,
            'ignoreParentInstanceSpec': True,
            'resources': patch_data
        }

    def make_patches(self):
        patches = []
        for component_id, parent_service in self.data.items():
            patch_dict = self._make_yappy_data(parent_service, self.resources)
            patches.append({'patch': patch_dict, 'component_id': component_id})
        return patches

    def _delete_existing(self, same_patches_pattern):
        existing_betas = self.session.post(YAPPY_LIST_BETAS_URL).json()

        for beta in existing_betas.get('objects', []):
            if same_patches_pattern in beta['name']:
                logging.info('Delete beta {}'.format(beta['name']))

                # Beta should be stopped before deleting
                stop_beta(beta['name'], self.token)
                resp = self.session.post(YAPPY_DELETE_BETA, data=json.dumps({'name': beta['name']}))
                logging.info('Deleting result: {}'.format(resp.text))

    def _create_beta_request(self):
        new_beta_name_suffix = '{}-{}-{}'.format(self.branch, self.tag, self.sandbox_task_id)
        if self.is_multicomponent():
            yappy_template_name = self.beta_type
        else:
            yappy_template_name = self.data.keys()[0]
        return {
            'template_name': yappy_template_name,
            'patches': self.make_patches(),
            'suffix': new_beta_name_suffix,
        }

    @retry(tries=5, delay=3)
    def create(self):
        if self.is_multicomponent():
            same_patches_pattern = '{}-{}-{}'.format(self.beta_type, self.branch, self.tag)
        else:
            same_patches_pattern = '{}-{}-{}'.format(YAPPY_DATA.get(self.beta_type), self.branch, self.tag)
        self._delete_existing(same_patches_pattern)
        payload = self._create_beta_request()
        logging.info('Payload:\n{}'.format(payload))

        response = self.session.post(YAPPY_CREATE_FROM_TEMPLATE_URL, data=json.dumps(payload))
        response_data = response.json()

        if not response_data.get('status') or response_data['status'] != 'SUCCESS':
            raise ValueError('Beta creation failed: {}\n{}'.format(response.status_code, response.text))

        while not self.check_status():
            logging.info('Waiting for beta %s creation' % self.name)
            time.sleep(60)

        logging.info('Beta created')
        self.service = self.get_beta_service()
        self.hamster_url = self.get_hamster_url()
        self.experimental_sources = self.get_experimental_sources()
        self.instances = self.get_instances()
        self.patch_file = payload['patches'][0]['patch']
        self.freeze()

    @retry(tries=5, delay=3)
    def get_beta_service(self):
        data = self.session.post(YAPPY_DATA_URL, data=json.dumps({'name': self.name})).json()
        if self.is_multicomponent():
            services = {}
            for component in data['components']:
                if component['templateId'] in self.data.keys():
                    services.update({component['templateId']: component['slot']['id']})
            return services
        return data['components'][0]['slot']['id']

    @retry(tries=5, delay=3)
    def get_hamster_url(self):
        data = self.session.post(YAPPY_DATA_URL, data=json.dumps({'name': self.name})).json()
        return 'https://{url}.ru'.format(url=data['domainNameWithoutTld'])

    @retry(tries=5, delay=3)
    def get_experimental_source(self, service, component=None):
        yp_endpoint_sets = self.yp_lite.get_endpoint_sets(service)
        logging.info('YP endpoint sets: %s', yp_endpoint_sets)
        if yp_endpoint_sets:
            return render_experimental_sources_from_yp(component or self.beta_type, yp_endpoint_sets)

        nanny_instances = self.nanny_cli.get_isolated_instances(service)
        logging.info('Nanny instances: %s', nanny_instances)
        if nanny_instances:
            return render_experimental_sources_from_nanny(self.beta_type, nanny_instances)

        return []

    def get_experimental_sources(self):
        if self.is_multicomponent():
            sources = {}
            for component, service in self.service.iteritems():
                sources.update({component: self.get_experimental_source(service, component)})
            return sources
        else:
            return self.get_experimental_source(self.service)

    def _extract_instances(self, endpoint_sets):
        instances = []
        for location, data in endpoint_sets.iteritems():
            instances.extend([endpoint['fqdn'] for endpoint in data[0]['status']['endpoints']])
        return instances

    @retry(tries=5, delay=3)
    def get_instances(self):
        if self.is_multicomponent():
            instances = {}
            for component, service in self.service.iteritems():
                endpoint_sets = self.yp_lite.get_endpoint_sets(service)
                logging.info('YP endpoint sets: %s', endpoint_sets)
                instances.update({component: self._extract_instances(endpoint_sets)})
            return instances
        endpoint_sets = self.yp_lite.get_endpoint_sets(self.service)
        return self._extract_instances(endpoint_sets)

    @retry(tries=5, delay=3)
    def check_status(self):
        data = self.session.post(YAPPY_STATUS_URL, data=json.dumps({'name': self.name})).json()
        if not data.get('state') or not data['state']['status']:
            logging.info('Check beta {} status failed:\n{}'.format(self.name, data))
            return False
        return data['state']['status'] == 'CONSISTENT'

    @retry(tries=5, delay=3)
    def freeze(self):
        data = {'beta_name': self.name, 'comment': 'freezed after creation'}
        resp = self.session.post(YAPPY_FREEZE_BETA, data=json.dumps(data))
        if resp.ok:
            logging.info('Beta {} freezed.'.format(self.name))
            logging.info('API response {}.'.format(resp.json()))
            return True
