# -*- coding: utf-8 -*-
import re
import logging
import requests
from urlparse import urljoin


GEOSEARCH_LOCATIONS = ['MAN', 'SAS', 'VLA']


def _get_shard_number(text):
    try:
        rgx = re.compile(r'\[\/labels\/shard_id\]\="(\d+)"')
        number = int(rgx.findall(text)[0])
        return '{number:0>3d}'.format(number=number - 1)
    except Exception:
        return None


class YpLightAPI(object):
    def __init__(self, oauth_token):
        # token is Nanny OAuth token, not YP
        self.url = 'https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/'
        self.oauth_token = oauth_token
        self.session = requests.Session()
        self.session.headers['Authorization'] = 'OAuth {}'.format(self.oauth_token)
        self.session.headers['Content-Type'] = 'application/json'

    def _get_full_url(self, uri):
        return urljoin(self.url, uri)

    def get_endpoint_sets(self, service):
        '''
        :returns: {cluster -> [EndpointSet]}
        '''
        url = self._get_full_url('endpoint-sets/ListEndpointSets/')
        endpoints = {}
        for location in GEOSEARCH_LOCATIONS:
            request_data = {
                'serviceId': service,
                'cluster': location
            }
            response = self.session.post(url, json=request_data)
            response.raise_for_status()
            result = response.json().get('endpointSets')
            if result:
                # If the service is not available in the location, the list is empty.
                endpoints[location] = result
        return endpoints

    def get_pods(self, service):
        '''
        :returns: [NYP.NClient.NApi.NProto.TPod]
        '''
        url = self._get_full_url('pod-sets/ListPods')
        pods = []
        for location in GEOSEARCH_LOCATIONS:
            request_data = {
                'serviceId': service,
                'cluster': location
            }
            response = self.session.post(url, json=request_data)
            logging.debug('{url} response code {code}'.format(url=url, code=response.status_code))
            logging.debug('Request data: {}'.format(request_data))
            logging.debug('Response data: {data}'.format(data=response.text))
            data = response.json()
            if data.get('pods'):
                pods.extend([pod['status']['dns']['persistentFqdn'] for pod in data.get('pods')])
        return pods
