# -*- coding: utf-8 -*
import time
from functools import wraps

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments
from sandbox.projects.common.nanny import nanny
import sandbox.projects.resource_types as rtypes
from sandbox.projects.geosearch.tools import stat
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosuggest import resources as geosuggest_resources
from sandbox.projects.common.geosearch.base_update import YtTablePath

from sandbox.projects.geosuggest.BuildGeosuggestToponymsData import BuildGeosuggestToponymsData
from sandbox.projects.geosuggest.BuildMapsGeoSuggestOrgMergedBin import BuildMapsGeoSuggestOrgMergedBin
from sandbox.projects.geosuggest.BuildMapsGeoSuggestOsmOrgMergedBin import BuildMapsGeoSuggestOsmOrgMergedBin
from sandbox.projects.geosuggest.BuildMapsGeoSuggestTransport import BuildMapsGeoSuggestTransport
from sandbox.projects.geosuggest.GeoSuggestAcceptance import GeoSuggestAcceptance
from sandbox.projects.geosuggest.GeoSuggestBuildSearchGroups import GeoSuggestBuildSearchGroups
from sandbox.projects.geosuggest.GeosuggestIndexPostprocessing import GeosuggestIndexPostprocessing
from sandbox.projects.geosuggest.GeosuggestPreparePOVTables import GeosuggestPreparePOVTables
from sandbox.projects.geosuggest.tests import ACCEPTANCE_AUTOTEST_PARAMS

from sandbox.projects.masstransit.resources import MASSTRANSIT_DATA_RASP
from sandbox.projects.geobase.Geodata5BinStable.resource import GEODATA5BIN_STABLE
from sandbox.projects.weather import WEATHER_SPECIAL_LOCATIONS


def push_to_stat(stage):
    def deco_stat(func):
        @wraps(func)
        def wrapper(self):
            stat_path = 'Maps_Plus_Beta/GeoSuggest/geosuggest_db_build'
            start_stage = 'start_' + stage
            finish_stage = 'finish_' + stage
            if not self.Context.stat.get(start_stage):
                self.Context.stat.update({start_stage: int(time.time())})
                self.Context.save()
            func(self)
            if not self.Context.stat.get(finish_stage):
                self.Context.stat.update({finish_stage: int(time.time())})
                self.Context.save()
                if self.Context.production_build:
                    stat.push_to_stat_table(stat_path,
                                            self.Context.start_time,
                                            start_stage,
                                            timestamp=self.Context.stat.get(start_stage))
                    stat.push_to_stat_table(stat_path,
                                            self.Context.start_time,
                                            finish_stage,
                                            timestamp=self.Context.stat.get(finish_stage))
        return wrapper
    return deco_stat


class BuildGeosuggestIndex(nanny.ReleaseToNannyTask2, sdk2.Task):
    '''
        Build geosuggest index
    '''

    class Parameters(sdk2.task.Parameters):
        kill_timeout = 43200
        geosuggest_data_builder = sdk2.parameters.Resource('GeoSuggest data builder executable',
                                                           resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_DATA_BUILDER)
        geosuggest_sandbox_binaries = sdk2.parameters.Resource('Geosuggest sandbox binaries pack',
                                                               resource_type=geosuggest_resources.GEO_SUGGEST_SANDBOX_BIN)
        suggest_data_builder = sdk2.parameters.Resource('Executable from quality/trailer/suggest/data_builder',
                                                        resource_type=geosuggest_resources.SUGGEST_DATA_BUILDER)
        normalize_requests_binary = sdk2.parameters.Resource('Request normalizer binary',
                                                             resource_type=rtypes.NORMALIZE_REQUESTS_EXECUTABLE)
        geobase = sdk2.parameters.Resource('Geodata5 binary',
                                           resource_type=GEODATA5BIN_STABLE)
        export_path = YtTablePath('Export YT path or export resource id',
                                  default_value='//home/altay/db/export/current-state/')
        yt_pool = sdk2.parameters.String('YT pool to use')
        yt_proxy = sdk2.parameters.String('YT proxy to use', default_value='hahn')
        with sdk2.parameters.Group('GeoSuggest transport build parameters') as transport_build_params:
            transport_routes_data = sdk2.parameters.Resource('Masstransit rasp data',
                                                             resource_type=MASSTRANSIT_DATA_RASP)
        with sdk2.parameters.Group('Merged orgs build parameters') as merged_orgs_build_params:
            orgs_weights_table = sdk2.parameters.Resource('YT table with weights',
                                                          resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_WEIGHTS_YT_TABLE)
            orgs_alt_weights_table = sdk2.parameters.String('Path to alternative YT table with weights only',
                                                            default_value='//home/qreg/geosuggest/production_data/static_weights/weights_orgs_region')
        with sdk2.parameters.Group('GeoSuggest search groups build parameters') as search_groups_build_params:
            geosuggest_search_groups_builder = sdk2.parameters.Resource('Search groups indexer',
                                                                        resource_type=geosuggest_resources.GEO_SUGGEST_SEARCH_GROUPS_BUILDER)
            altay_snapshot = YtTablePath('Altay snapshot',
                                         default_value='//home/altay/db/export/current-state/snapshot/company')
            toponym_org_duplicates = sdk2.parameters.String('Toponym org duplicates',
                                                            default_value='//home/altay/toponyms/current-state/mapping')
        with sdk2.parameters.Group('Geocoder YT export build parameters') as geocoder_yt_export_build_params:
            geocoder_export_path = sdk2.parameters.String('Geocoder export YT path')
            geocoder_osm_export_path = sdk2.parameters.String('Geocoder OSM export YT path')
            toponyms_global_factors_table_path = sdk2.parameters.String('Toponyms global factors table path',
                                                                        default_value='//home/qreg/geosuggest/production_data/doc_factors/clicks_and_shows')
            toponyms_weights_table_path = sdk2.parameters.String('Toponyms weights table path',
                                                                 default_value='//home/qreg/geosuggest/prepared/toponym_weights')
        with sdk2.parameters.Group('Merged OSM orgs build parameters') as merged_osm_orgs_build_params:
            osm_yt_exported_orgs_root = sdk2.parameters.String('Path to alternative YT directory with exported OSM orgs')
            osm_weights_table = sdk2.parameters.Resource('YT table with OSM weights',
                                                         resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_WEIGHTS_OSM_YT_TABLE)
            osm_alt_weights_table = sdk2.parameters.String('Path to alternative YT table with weights only',
                                                           default_value='//home/qreg/geosuggest/production_data/static_weights/weights_orgs_region')
            acceptable_country_codes = sdk2.parameters.List('Country codes to use in merged data (empty list for all countries)')
        with sdk2.parameters.Group('Weather parameters') as weather_params:
            weather_special_locations = sdk2.parameters.Resource('special_locations.json',
                                                                 resource_type=WEATHER_SPECIAL_LOCATIONS)
        with sdk2.parameters.Group('Pack production data parameters') as pack_production_data_params:
            prefixtops_full = sdk2.parameters.Resource('Full prefixtop',
                                                       resource_type=geosuggest_resources.GEO_SUGGEST_PREFIXTOPS)
            prefixtops_taxi_world = sdk2.parameters.Resource('Taxi prefixtop for new countries',
                                                             resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_PREFIXTOPS_TAXI_WORLD)
            search_history_bin = sdk2.parameters.Resource('geo_search_dictionary/bin',
                                                          resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_GEO_SEARCH_HISTORY_BIN)
            geosuggest_static_data = sdk2.parameters.Resource('GeoSuggest daemon static data (format is identical to GEO_SUGGEST_DATA)',
                                                              resource_type=geosuggest_resources.GEO_SUGGEST_STATIC_DATA)
            geosearch_ngrams = sdk2.parameters.Resource('geo_search ngrams (compiled trie)',
                                                        resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_GEO_SEARCH_NGRAMS_BIN)
            misspells_tries = sdk2.parameters.Resource('Misspells tries',
                                                       resource_type=geosuggest_resources.MAPS_GEO_SUGGEST_MISSPELLS)
            geosuggest_top_rubrics_by_regions = sdk2.parameters.Resource('Top rubrics by regions',
                                                                         resource_type=geosuggest_resources.GEO_SUGGEST_TOP_RUBRICS_BY_REGIONS_TSV)
            geosuggest_top_rubrics_config = sdk2.parameters.Resource('Top rubrics config',
                                                                     resource_type=geosuggest_resources.GEO_SUGGEST_TOP_RUBRICS_CONFIG_JSON)
            foreign_trie = sdk2.parameters.Resource('Foreign trie',
                                                       resource_type=geosuggest_resources.GEO_SUGGEST_FOREIGN_TRIE)

        with sdk2.parameters.Group('Acceptance parameters') as acceptance_params:
            geosuggestd = sdk2.parameters.Resource('Geo suggest daemon',
                                                   resource_type=geosuggest_resources.GEO_SUGGEST_WEBDAEMON)

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        environments = (environments.PipEnvironment('yandex-yt', use_wheel=True),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def _wait(self, tasks):
        raise sdk2.WaitTask(tasks,
                            ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                            wait_all=True)

    def _check(self, tasks):
        for task_id in tasks:
            task = sdk2.Task[task_id]
            if task.status in self.bad_statuses:
                raise SandboxTaskFailureError('Task #{tid} {task_type} failed'.format(tid=task.id, task_type=task.type))

    def resolve_yt_symlink(self, path):
        import yt.wrapper as yt
        yt.config['token'] = sdk2.Vault.data('GEO_SUGGEST', 'yt_token')
        yt.config['proxy']['url'] = 'hahn.yt.yandex.net'
        try:
            return yt.get(path + '&/@target_path')
        except yt.YtHttpResponseError:
            return path

    def pre_run(self):
        with self.memoize_stage.PRE_RUN(commit_on_entrance=False):
            user_whitelist = ['robot-qreg']
            self.Context.production_build = self.parent and self.author in user_whitelist
            self.Context.start_time = self.created.strftime('geosuggest-%Y-%m-%dT%H:%M:%S+appendix')
            self.Context.stat = {}
            self.bad_statuses = ['FAILURE',
                                 'EXCEPTION',
                                 'TIMEOUT']

    def make_prepare_pov_tables_task(self, pov):
        prepare_pov_tables_task = GeosuggestPreparePOVTables(
            self,
            owner=self.owner,
            description='Preparation of {} pov tables'.format(pov),
            create_sub_task=True,
            geosuggest_binaries=int(self.Parameters.geosuggest_sandbox_binaries),
            geocoder_export=self.resolve_yt_symlink(self.Parameters.geocoder_export_path),
            pov=pov,
            output_dir='//home/qreg/geosuggest/prepared/toponym_index',
            ttl_days=7,
            yt_cluster=self.Parameters.yt_proxy,
            yt_pool=self.Parameters.yt_pool
        )
        prepare_pov_tables_task.enqueue()
        return prepare_pov_tables_task

    @push_to_stat('prepare_pov_tables')
    def prepare_pov_tables(self):
        with self.memoize_stage.PREPARE_POV_TABLES(commit_on_entrance=False):
            tasks = []

            if not self.Context.prepare_ru_pov_tables_task:
                task = self.make_prepare_pov_tables_task(pov='RU')
                self.Context.prepare_ru_pov_tables_task = task.id
                tasks.append(task)
            if not self.Context.prepare_ua_pov_tables_task:
                task = self.make_prepare_pov_tables_task(pov='UA')
                self.Context.prepare_ua_pov_tables_task = task.id
                tasks.append(task)
            self._wait(tasks)

            self._check([self.Context.prepare_ru_pov_tables_task, self.Context.prepare_ua_pov_tables_task])

    def build_transport_data(self):
        if not self.Context.build_transport_task:
            build_transport_task_class = sdk2.Task[BuildMapsGeoSuggestTransport.type]
            build_transport_task = build_transport_task_class(build_transport_task_class.current,
                                                              owner=self.owner,
                                                              data_builder=int(self.Parameters.geosuggest_data_builder),
                                                              transport=int(self.Parameters.transport_routes_data),
                                                              kill_timeout=self.Parameters.kill_timeout,
                                                              create_sub_task=True)
            build_transport_task.save()
            build_transport_task.enqueue()
            self.Context.build_transport_task = build_transport_task.id
            return build_transport_task

    def build_merged_orgs(self):
        if not self.Context.build_merged_orgs_task:
            build_merged_orgs_task_class = sdk2.Task[BuildMapsGeoSuggestOrgMergedBin.type]
            build_merged_orgs_task = build_merged_orgs_task_class(build_merged_orgs_task_class.current,
                                                                  owner=self.owner,
                                                                  data_builder=int(self.Parameters.geosuggest_data_builder),
                                                                  geodata5_bin_stable=int(self.Parameters.geobase),
                                                                  geosuggest_sandbox_bin=int(self.Parameters.geosuggest_sandbox_binaries),
                                                                  geosuggest_sandbox_vault_name='yt_token',
                                                                  geosuggest_sandbox_vault_owner='GEO_SUGGEST',
                                                                  geosuggest_yt_pool=self.Parameters.yt_pool,
                                                                  geosuggest_yt_proxy=self.Parameters.yt_proxy,
                                                                  suggest_builder=int(self.Parameters.suggest_data_builder),
                                                                  weight_info2=self.Parameters.orgs_alt_weights_table,
                                                                  weights_yt_table=int(self.Parameters.orgs_weights_table),
                                                                  kill_timeout=self.Parameters.kill_timeout,
                                                                  create_sub_task=True)
            build_merged_orgs_task.save()
            build_merged_orgs_task.enqueue()
            self.Context.build_merged_orgs_task = build_merged_orgs_task.id
            return build_merged_orgs_task

    def build_geosuggest_search_groups(self):
        if not self.Context.build_geosuggest_search_groups_task:
            build_geosuggest_search_groups_task_class = sdk2.Task[GeoSuggestBuildSearchGroups.type]
            build_geosuggest_search_groups_task = build_geosuggest_search_groups_task_class(
                build_geosuggest_search_groups_task_class.current,
                owner=self.owner,
                geobase_snapshot=self.Parameters.altay_snapshot,
                geosuggest_sandbox_vault_name='yt_token',
                geosuggest_sandbox_vault_owner='GEO_SUGGEST',
                geosuggest_search_groups_builder=int(self.Parameters.geosuggest_search_groups_builder),
                toponym_org_duplicates=self.Parameters.toponym_org_duplicates,
                kill_timeout=self.Parameters.kill_timeout,
                create_sub_task=True,
                geosuggest_yt_pool=self.Parameters.yt_pool,
                geosuggest_yt_proxy=self.Parameters.yt_proxy)
            build_geosuggest_search_groups_task.save()
            build_geosuggest_search_groups_task.enqueue()
            self.Context.build_geosuggest_search_groups_task = build_geosuggest_search_groups_task.id
            return build_geosuggest_search_groups_task

    def make_geocoder_export_task(self, geocoder_export_path, description, custom_pov_path=None):
        task = BuildGeosuggestToponymsData(
            self,
            description=description,
            kill_timeout=self.Parameters.kill_timeout,
            owner=self.owner,
            geosuggest_builder=int(self.Parameters.geosuggest_sandbox_binaries),
            yt_cluster=self.Parameters.yt_proxy,
            yt_pool=self.Parameters.yt_pool,
            geocoder_export_path=self.resolve_yt_symlink(geocoder_export_path),
            global_factors_table_path=self.Parameters.toponyms_global_factors_table_path,
            weights_table_path=self.Parameters.toponyms_weights_table_path,
            output_dir_path='//home/qreg/geosuggest/prepared/toponym_index',
            ttl_days=7,
            create_sub_task=True,
            custom_pov_path=custom_pov_path
        )
        task.save()
        task.enqueue()
        return task

    def build_geocoder_export(self):
        if not self.Context.build_geocoder_export_task:
            prepare_pov_tables_task = sdk2.Task[self.Context.prepare_ru_pov_tables_task]
            prepared_pov_tables = prepare_pov_tables_task.Context.prepared_pov_tables
            task = self.make_geocoder_export_task(
                geocoder_export_path=self.Parameters.geocoder_export_path,
                description='Build toponyms from RU POV',
                custom_pov_path=prepared_pov_tables)
            self.Context.build_geocoder_export_task = task.id
            return task

    def build_geocoder_export_ua(self):
        if not self.Context.build_geocoder_export_ua_task:
            prepare_pov_tables_task = sdk2.Task[self.Context.prepare_ua_pov_tables_task]
            prepared_pov_tables = prepare_pov_tables_task.Context.prepared_pov_tables
            task = self.make_geocoder_export_task(
                geocoder_export_path=self.Parameters.geocoder_export_path,
                description='Build toponyms from UA POV',
                custom_pov_path=prepared_pov_tables)
            self.Context.build_geocoder_export_ua_task = task.id
            return task

    def build_geocoder_osm_export(self, geocoder_osm_export_path):
        if not self.Context.build_geocoder_osm_export_task:
            task = self.make_geocoder_export_task(
                geocoder_export_path=geocoder_osm_export_path,
                description='Build toponyms from OSM')
            self.Context.build_geocoder_osm_export_task = task.id
            return task

    def build_merged_osm_orgs(self):
        if not self.Context.build_merged_osm_orgs_task:
            build_merged_osm_orgs_task_class = sdk2.Task[BuildMapsGeoSuggestOsmOrgMergedBin.type]

            osm_yt_exported_orgs_root = self.Parameters.osm_yt_exported_orgs_root
            if not osm_yt_exported_orgs_root:
                osm_yt_exported_orgs_root = None

            acceptable_country_codes = self.Parameters.acceptable_country_codes
            acceptable_country_codes = ' '.join(acceptable_country_codes)

            build_merged_osm_orgs_task = build_merged_osm_orgs_task_class(
                build_merged_osm_orgs_task_class.current,
                owner=self.owner,
                data_builder=int(self.Parameters.geosuggest_data_builder),
                geodata5_bin_stable=int(self.Parameters.geobase),
                geosuggest_sandbox_bin=int(self.Parameters.geosuggest_sandbox_binaries),
                geosuggest_sandbox_vault_name='yt_token',
                geosuggest_sandbox_vault_owner='GEO_SUGGEST',
                geosuggest_yt_pool=self.Parameters.yt_pool,
                geosuggest_yt_proxy=self.Parameters.yt_proxy,
                suggest_builder=int(self.Parameters.suggest_data_builder),
                weight_info2=self.Parameters.osm_alt_weights_table,
                weights_yt_table=int(self.Parameters.osm_weights_table),
                kill_timeout=self.Parameters.kill_timeout,
                create_sub_task=True,
                yt_exported_orgs_root=osm_yt_exported_orgs_root,
                acceptable_country_codes=acceptable_country_codes)
            build_merged_osm_orgs_task.save()
            build_merged_osm_orgs_task.enqueue()
            self.Context.build_merged_osm_orgs_task = build_merged_osm_orgs_task.id
            return build_merged_osm_orgs_task

    @push_to_stat('prepare_data')
    def prepare_data(self):
        with self.memoize_stage.RUN_SUBTASKS(commit_on_entrance=False):
            subtasks = []
            subtasks.append(self.build_transport_data())
            subtasks.append(self.build_merged_orgs())
            subtasks.append(self.build_geosuggest_search_groups())
            subtasks.append(self.build_geocoder_export())
            subtasks.append(self.build_geocoder_export_ua())
            if self.Parameters.geocoder_osm_export_path:
                task = self.build_geocoder_osm_export(
                    geocoder_osm_export_path=self.Parameters.geocoder_osm_export_path)
                subtasks.append(task)
            subtasks.append(self.build_merged_osm_orgs())
            self._wait(subtasks)
            self._check([t.id for t in subtasks])

    def get_internal_resources(self):
        merged_orgs_task = sdk2.Task[self.Context.build_merged_orgs_task]
        merged_osm_orgs_task = sdk2.Task[self.Context.build_merged_osm_orgs_task]
        search_groups_task = sdk2.Task[self.Context.build_geosuggest_search_groups_task]
        transport_data_task = sdk2.Task[self.Context.build_transport_task]
        toponyms_export_build_task = sdk2.Task[self.Context.build_geocoder_export_task]
        toponyms_export_ua_build_task = sdk2.Task[self.Context.build_geocoder_export_ua_task]

        prepare_ru_pov_tables_task = sdk2.Task[self.Context.prepare_ru_pov_tables_task]
        prepare_ua_pov_tables_task = sdk2.Task[self.Context.prepare_ua_pov_tables_task]

        resources = {
            'merged_orgs': int(sdk2.Resource[geosuggest_resources.MAPS_GEO_SUGGEST_ORG_MERGED_BIN].find(task=merged_orgs_task).first()),
            'merged_osm_orgs': int(sdk2.Resource[geosuggest_resources.MAPS_GEO_SUGGEST_OSM_ORG_MERGED_BIN].find(task=merged_osm_orgs_task).first()),
            'search_groups': int(sdk2.Resource[geosuggest_resources.MAPS_GEO_SUGGEST_SEARCH_GROUPS].find(task=search_groups_task).first()),
            'transport_data': int(sdk2.Resource[geosuggest_resources.MAPS_GEO_SUGGEST_TRANSPORT].find(task=transport_data_task).first()),
            'toponyms_export_data': toponyms_export_build_task.Context.toponyms_data_table,
            'toponyms_export_ua_data': toponyms_export_ua_build_task.Context.toponyms_data_table,
            'toponyms_weights_table': toponyms_export_build_task.Context.org2addr_data_table,
            'prepared_ru_pov_tables': prepare_ru_pov_tables_task.Context.prepared_pov_tables,
            'prepared_ua_pov_tables': prepare_ua_pov_tables_task.Context.prepared_pov_tables
        }
        if self.Context.build_geocoder_osm_export_task:
            task = sdk2.Task[self.Context.build_geocoder_osm_export_task]
            resources['toponyms_osm_export_data'] = task.Context.toponyms_data_table
        return resources

    @push_to_stat('index_postprocessing')
    def index_postprocessing(self):
        with self.memoize_stage.POSTPROCESSING(commit_on_entrance=False):
            if not self.Context.postprocessing_task:
                internal_resources = self.get_internal_resources()
                index_postprocessing_task = GeosuggestIndexPostprocessing(
                    self,
                    geosuggest_static_data=int(self.Parameters.geosuggest_static_data),
                    geodata5bin_stable=int(self.Parameters.geobase),
                    toponyms_export_data=internal_resources.get('toponyms_export_data'),
                    toponyms_osm_export_data=internal_resources.get('toponyms_osm_export_data'),
                    toponyms_export_ua_data=internal_resources.get('toponyms_export_ua_data'),
                    prepared_ru_pov_tables=internal_resources.get('prepared_ru_pov_tables'),
                    prepared_ua_pov_tables=internal_resources.get('prepared_ua_pov_tables'),
                    altay_export=self.Parameters.export_path,
                    org2addr_toponyms_table=internal_resources.get('toponyms_weights_table'),
                    geosuggest_misspells=int(self.Parameters.misspells_tries),
                    geosuggest_org_2_addr=internal_resources.get('org_2_addr'),
                    geosuggest_org_merged=internal_resources.get('merged_orgs'),
                    geosuggest_osm_org_merged=internal_resources.get('merged_osm_orgs'),
                    geosuggest_search_groups=internal_resources.get('search_groups'),
                    geosuggest_transport=internal_resources.get('transport_data'),
                    geosuggest_weather_special_locations=self.Parameters.weather_special_locations,
                    prefixtops_full=int(self.Parameters.prefixtops_full),
                    prefixtops_taxi_world=int(self.Parameters.prefixtops_taxi_world),
                    search_history_bin=int(self.Parameters.search_history_bin),
                    geosearch_ngrams=int(self.Parameters.geosearch_ngrams),
                    geosuggest_sandbox_binaries=int(self.Parameters.geosuggest_sandbox_binaries),
                    geosuggest_top_rubrics_by_regions=int(self.Parameters.geosuggest_top_rubrics_by_regions),
                    geosuggest_top_rubrics_config=int(self.Parameters.geosuggest_top_rubrics_config or 0),
                    owner=self.owner,
                    kill_timeout=self.Parameters.kill_timeout,
                    create_sub_task=True,
                    foreign_trie=int(self.Parameters.foreign_trie),
                    yt_pool=self.Parameters.yt_pool,
                    yt_proxy=self.Parameters.yt_proxy
                )
                index_postprocessing_task.enqueue()
                self.Context.postprocessing_task = index_postprocessing_task.id
                self._wait([index_postprocessing_task])
            self._check([self.Context.postprocessing_task])

    def aceptance(self):
        with self.memoize_stage.ACCEPTANCE(commit_on_entrance=False):
            if not self.Context.acceptance_task:
                postprocessing_task = sdk2.Task[self.Context.postprocessing_task]
                data_resource = int(sdk2.Resource[geosuggest_resources.GEO_SUGGEST_DATA].find(task=postprocessing_task).first())
                dolbilka_plan_resource = int(sdk2.Resource[geosuggest_resources.GEO_SUGGEST_WEBDAEMON_PLAN].find().first())
                acceptance_task = GeoSuggestAcceptance(self,
                                                       data_builder=int(self.Parameters.geosuggest_data_builder),
                                                       geosuggest_autotest_arguments=' '.join(ACCEPTANCE_AUTOTEST_PARAMS),
                                                       geosuggest_use_personal_mocks=True,
                                                       geosuggest_data_resource_id=data_resource,
                                                       geosuggest_dolbilka_plan_resource_id=dolbilka_plan_resource,
                                                       geosuggest_emails_do_not_send=False,
                                                       geosuggest_emails_to='geo-suggest@yandex-team.ru',
                                                       geosuggest_external_daemon_url='http://suggest-maps.yandex.{domain}/suggest-geo',
                                                       geosuggest_path_to_autotest='autotest',
                                                       geosuggest_run_performance_tests=True,
                                                       geosuggest_sandbox_vault_nanny_token_name='nanny_token',
                                                       geosuggest_sandbox_vault_owner='GEO_SUGGEST',
                                                       geosuggest_shutdown_timeout=120,
                                                       geosuggest_stable_result_parameter='production_snapshot',
                                                       geosuggest_start_timeout=7200,
                                                       geosuggest_sandbox_binaries=int(self.Parameters.geosuggest_sandbox_binaries),
                                                       geosuggestd_resource_id=int(self.Parameters.geosuggestd),
                                                       isolated_mode=False,
                                                       owner=self.owner,
                                                       kill_timeout=self.Parameters.kill_timeout,
                                                       create_sub_task=True)
                acceptance_task.enqueue()
                self.Context.acceptance_task = acceptance_task.id

    def on_execute(self):
        self.pre_run()
        self.prepare_pov_tables()
        self.prepare_data()
        self.index_postprocessing()
        self.aceptance()

    def on_release(self, params):
        self.server.release(task_id=self.Context.postprocessing_task,
                            type=params['release_status'],
                            subject=params['release_subject'],
                            comments=params['release_comments'])
        self.server.release(task_id=self.parent.id,
                            type=params['release_status'],
                            subject=params['release_subject'],
                            comments=params['release_comments'])
