# -*- coding: utf-8 -*

import os
import shutil
import logging
import subprocess

from sandbox import sdk2
from sandbox.sandboxsdk.paths import get_logs_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.projects.geosuggest import resources


class BuildGeosuggestOrg2Addr(sdk2.Task):
    '''
        Merges permalinks with toponym ids
    '''

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.task.Parameters):
        kill_timeout = 4 * 60 * 60
        binary_res = sdk2.parameters.Resource('Geosuggest builder binaries',
                                              resource_type=resources.GEO_SUGGEST_SANDBOX_BIN)
        altay_export = sdk2.parameters.String('Altay export YT path',
                                              required=True)
        org2addr_toponyms_table = sdk2.parameters.String('Toponyms table YT path',
                                                         required=True)
        weights_table = sdk2.parameters.String('Weights table path',
                                               default_value='//home/qreg/geosuggest/prepared/org2addr_weights')

    def run(self, binary_path):
        proc_env = os.environ.copy()
        self.yt_token = sdk2.Vault.data('GEO_SUGGEST', 'yt_token')
        proc_env['YT_TOKEN'] = self.yt_token
        merged_companies_path = './oid2addr.tsv'
        not_merged_companies_path = './companies2.not_joined'
        cmd = [
            binary_path,
            'oid2addr',
            '-s', 'hahn',
            '-c', os.path.join(self.Parameters.altay_export, 'snapshot/company'),
            '-w', self.Parameters.weights_table,
            '-t', self.Parameters.org2addr_toponyms_table,
            '-m', merged_companies_path,
            '-n', not_merged_companies_path
        ]
        logging.info('Running: {}'.format(cmd))
        log_file_path = os.path.join(get_logs_folder(), 'toponyms_data_prepare.log')
        with open(log_file_path, 'w') as log_file:
            try:
                subprocess.check_call(cmd,
                                      env=proc_env,
                                      stdout=log_file,
                                      stderr=subprocess.STDOUT)
                return merged_companies_path, not_merged_companies_path
            except subprocess.CalledProcessError:
                logging.exception('%s command failed' % cmd)
                raise SandboxTaskFailureError('Building oid2addr failed')

    def create_resource(self, file_name, dir_name, resource_type, description):
        os.makedirs(dir_name)
        new_path = os.path.join(dir_name, file_name)
        shutil.copyfile(file_name, new_path)
        resource = sdk2.Resource[resource_type]
        current_resource = resource(
            self,
            description,
            new_path.split('/')[0]
        )
        data = sdk2.ResourceData(current_resource)
        data.ready()

    def on_execute(self):
        binary_path = str(sdk2.ResourceData(self.Parameters.binary_res).path) + '/suggest_prepare_data'
        merged_companies_path, not_merged_companies_path = self.run(binary_path)

        self.create_resource(
            merged_companies_path,
            'oid2addr/full',
            resources.MAPS_GEO_SUGGEST_ORG_2_ADDR,
            'org2addr'
        )
        self.create_resource(
            not_merged_companies_path,
            'not_joined/full',
            resources.MAPS_GEO_SUGGEST_ORG_2_ADDR_NOT_JOINED,
            'companies2.not_joined'
        )
