import os
import sys

from sandbox.sandboxsdk.parameters import LastReleasedResource, SandboxStringParameter
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.geosuggest import resources
from sandbox.projects.geosuggest.common import qp
from sandbox.projects.geosuggest.common.parameters import (GeoSuggestDataBuilderParameter, AdditionalEnvParameter, )
from sandbox.projects.common.utils import get_or_default


class GeoSuggestWeightsDumpParameter(LastReleasedResource):
    name = 'geosuggest_weights_dump'
    description = 'Dump from YT with weights'
    required = True
    resource_type = resources.MAPS_GEO_SUGGEST_WEIGHTS_DUMP


class GeoSuggestWeightFieldParameter(SandboxStringParameter):
    name = 'geosuggest_weight_field'
    description = 'Name of field in YT table with weight'
    required = True


class BuildMapsGeoSuggestWeights(qp.GeoSuggestQPTask):

    type = 'BUILD_MAPS_GEO_SUGGEST_WEIGHTS'
    input_parameters = [
        GeoSuggestDataBuilderParameter,
        GeoSuggestWeightsDumpParameter,
        GeoSuggestWeightFieldParameter,
        AdditionalEnvParameter,
    ]

    TIMEOUT = 10 * 3600
    execution_space = 800 * 1024
    disk_space = 800 * 1024

    def on_execute(self):
        AdditionalEnvParameter.ApplyToEnviron(self.ctx)

        data_builder_dir = self.sync_resource(
            get_or_default(self.ctx, GeoSuggestDataBuilderParameter),
        )
        convert_yt_weights_script_path = os.path.join(data_builder_dir, 'weights', 'convert_yt_weights.py')
        dump_path = self.sync_resource(get_or_default(self.ctx, GeoSuggestWeightsDumpParameter))
        output_dir = os.path.join(self.abs_path(), 'weights')
        run_process(['mkdir', output_dir])

        run_process([
            sys.executable,
            convert_yt_weights_script_path,
            '--input', dump_path,
            '--output', output_dir,
            '--weight_field', get_or_default(self.ctx, GeoSuggestWeightFieldParameter),
        ], log_prefix='convert_yt_weights')

        self.create_resource(
            'Geo suggest geo weights',
            os.path.join(output_dir, 'weight_info.geo.txt'),
            resources.MAPS_GEO_SUGGEST_GEO_WEIGHT_INFO,
        )
        self.create_resource(
            'Geo suggest org weights',
            os.path.join(output_dir, 'weight_info.org.txt'),
            resources.MAPS_GEO_SUGGEST_ORG_WEIGHT_INFO,
        )

    @classmethod
    def on_qp_tests(cls, task, rt, prev, curr, env):
        if rt == resources.MAPS_GEO_SUGGEST_GEO_WEIGHT_INFO:
            qp.check_dir_size_percent(env, prev, curr, 20)
        elif rt == resources.MAPS_GEO_SUGGEST_ORG_WEIGHT_INFO:
            qp.check_dir_size_percent(env, prev, curr, 20)


__Task__ = BuildMapsGeoSuggestWeights
